/*
 * Decompiled with CFR 0.152.
 */
package io.crate.blob;

import io.crate.blob.BlobContainer;
import io.crate.blob.BlobWriteException;
import io.crate.blob.exceptions.BlobAlreadyExistsException;
import io.crate.blob.exceptions.DigestMismatchException;
import io.crate.common.Hex;
import io.netty.buffer.ByteBuf;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.transport.netty4.Netty4Utils;

public class DigestBlob
implements Closeable {
    private final String digest;
    private final BlobContainer container;
    private final UUID transferId;
    protected File file;
    private FileChannel fileChannel;
    private FileChannel headFileChannel;
    private int size;
    private long headLength;
    private AtomicLong headSize;
    private MessageDigest md;
    private long chunks;
    private CountDownLatch headCatchedUpLatch;
    private static final Logger LOGGER = LogManager.getLogger(DigestBlob.class);

    public DigestBlob(BlobContainer container, String digest, UUID transferId) {
        this.digest = digest;
        this.container = container;
        this.size = 0;
        this.transferId = transferId;
    }

    public String getDigest() {
        return this.digest;
    }

    public int size() {
        return this.size;
    }

    public File file() {
        return this.file;
    }

    private static Path getTmpFilePath(BlobContainer blobContainer, String digest, UUID transferId) {
        return blobContainer.getTmpDirectory().resolve(digest + "." + transferId.toString());
    }

    private File createTmpFile() throws IOException {
        File tmpFile = DigestBlob.getTmpFilePath(this.container, this.digest, this.transferId).toFile();
        tmpFile.createNewFile();
        return tmpFile;
    }

    private void updateDigest(ByteBuffer bbf) throws IOException {
        if (this.md == null) {
            try {
                this.md = MessageDigest.getInstance("SHA-1");
            }
            catch (NoSuchAlgorithmException e) {
                throw new IOException(e);
            }
        }
        this.md.update(bbf.slice());
    }

    private void addContent(ByteBuf buffer, boolean last) throws IOException {
        if (buffer != null) {
            int readableBytes = buffer.readableBytes();
            ByteBuffer byteBuffer = buffer.nioBuffer();
            if (this.file == null) {
                this.file = this.createTmpFile();
            }
            if (this.fileChannel == null) {
                FileOutputStream outputStream = new FileOutputStream(this.file);
                this.fileChannel = outputStream.getChannel();
            }
            int written = 0;
            do {
                if (this.headLength != 0L) continue;
                this.updateDigest(byteBuffer);
            } while ((written += this.fileChannel.write(byteBuffer)) < readableBytes);
            this.size += readableBytes;
            buffer.readerIndex(buffer.readerIndex() + written);
            ++this.chunks;
        }
        if (last) {
            if (this.file == null) {
                this.file = this.createTmpFile();
            }
            if (this.fileChannel == null) {
                FileOutputStream outputStream = new FileOutputStream(this.file);
                this.fileChannel = outputStream.getChannel();
            }
            this.fileChannel.force(false);
            this.fileChannel.close();
            this.fileChannel = null;
        } else if (buffer == null) {
            throw new NullPointerException("buffer");
        }
    }

    private void calculateDigest() {
        assert (this.headSize.get() == this.headLength) : "Head hasn't catched up, can't calculate digest";
        try (FileInputStream stream = new FileInputStream(this.file);){
            int bytesRead;
            stream.skipNBytes(this.headLength);
            byte[] buffer = new byte[4096];
            while ((bytesRead = stream.read(buffer, 0, 4096)) > 0) {
                this.md.update(buffer, 0, bytesRead);
            }
        }
        catch (IOException ex) {
            LOGGER.error("error accessing file to calculate digest", (Throwable)ex);
        }
    }

    public File commit() throws DigestMismatchException, BlobAlreadyExistsException {
        if (this.headLength > 0L) {
            this.calculateDigest();
        }
        assert (this.md != null) : "MessageDigest should not be null";
        try {
            String contentDigest = Hex.encodeHexString((byte[])this.md.digest());
            if (!contentDigest.equals(this.digest)) {
                this.file.delete();
                throw new DigestMismatchException(this.digest, contentDigest);
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.headFileChannel});
            this.headFileChannel = null;
            throw throwable;
        }
        IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.headFileChannel});
        this.headFileChannel = null;
        File newFile = this.container.getFile(this.digest);
        Semaphore semaphore = this.container.digestCoordinator(this.digest);
        try {
            semaphore.acquire();
            try {
                if (Files.exists(newFile.toPath(), new LinkOption[0])) {
                    throw new BlobAlreadyExistsException(this.digest);
                }
                this.file.renameTo(newFile);
                this.file = null;
            }
            finally {
                semaphore.release();
            }
        }
        catch (InterruptedException e) {
            LOGGER.error("Unable to commit blob {}", (Object)e, (Object)this.file.getName());
            throw new IllegalStateException("Unable to commit blob because exclusive execution could not be achieved");
        }
        return newFile;
    }

    public File getContainerFile() {
        return this.container.getFile(this.digest);
    }

    public void addContent(BytesReference content, boolean last) {
        try {
            this.addContent(Netty4Utils.toByteBuf(content), last);
        }
        catch (IOException e) {
            throw new BlobWriteException(this.digest, this.size, e);
        }
    }

    public void addToHead(BytesReference content) throws IOException {
        int written;
        if (content == null) {
            return;
        }
        ByteBuf byteBuf = Netty4Utils.toByteBuf(content);
        int readableBytes = byteBuf.readableBytes();
        assert ((long)readableBytes + this.headSize.get() <= this.headLength) : "Got too many bytes in addToHead()";
        ByteBuffer byteBuffer = byteBuf.nioBuffer();
        for (written = 0; written < readableBytes; written += this.headFileChannel.write(byteBuffer)) {
            this.updateDigest(byteBuffer);
        }
        this.headSize.addAndGet(written);
        if (this.headSize.get() == this.headLength) {
            this.headCatchedUpLatch.countDown();
        }
    }

    public long chunks() {
        return this.chunks;
    }

    public static DigestBlob resumeTransfer(BlobContainer blobContainer, String digest, UUID transferId, long currentPos) {
        DigestBlob digestBlob = new DigestBlob(blobContainer, digest, transferId);
        digestBlob.file = DigestBlob.getTmpFilePath(blobContainer, digest, transferId).toFile();
        try {
            LOGGER.trace("Resuming DigestBlob {}. CurrentPos {}", (Object)digest, (Object)currentPos);
            digestBlob.headFileChannel = new FileOutputStream(digestBlob.file, false).getChannel();
            digestBlob.headLength = currentPos;
            digestBlob.headSize = new AtomicLong();
            digestBlob.headCatchedUpLatch = new CountDownLatch(1);
            RandomAccessFile raf = new RandomAccessFile(digestBlob.file, "rw");
            raf.setLength(currentPos);
            raf.close();
            FileOutputStream outputStream = new FileOutputStream(digestBlob.file, true);
            digestBlob.fileChannel = outputStream.getChannel();
        }
        catch (IOException ex) {
            LOGGER.error("error resuming transfer of {}, id: {}", (Object)ex, (Object)digest, (Object)transferId);
            return null;
        }
        return digestBlob;
    }

    public void waitForHead() {
        if (this.headLength == 0L) {
            return;
        }
        assert (this.headCatchedUpLatch != null) : "headCatchedUpLatch should not be null";
        try {
            this.headCatchedUpLatch.await();
        }
        catch (InterruptedException e) {
            Thread.interrupted();
        }
    }

    @Override
    public void close() throws IOException {
        if (this.file != null) {
            this.file.delete();
        }
    }
}

