/*
 * Decompiled with CFR 0.152.
 */
package io.crate.types;

import io.crate.Streamer;
import io.crate.sql.tree.ColumnDefinition;
import io.crate.sql.tree.ColumnType;
import io.crate.sql.tree.Expression;
import io.crate.types.DataType;
import io.crate.types.DataTypes;
import io.crate.types.NumericStorage;
import io.crate.types.StorageSupport;
import io.crate.types.TypeSignature;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import org.apache.lucene.util.RamUsageEstimator;
import org.elasticsearch.Version;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class NumericType
extends DataType<BigDecimal>
implements Streamer<BigDecimal> {
    public static final int ID = 22;
    public static final String NAME = "numeric";
    public static final NumericType INSTANCE = new NumericType(null, null);
    @Nullable
    private final Integer scale;
    @Nullable
    private final Integer precision;
    private final MathContext mathContext;

    public static DataType<?> of(List<Integer> parameters) {
        if (parameters.isEmpty() || parameters.size() > 2) {
            throw new IllegalArgumentException("The numeric type support one or two parameter arguments, received: " + parameters.size());
        }
        if (parameters.size() == 1) {
            return new NumericType(parameters.get(0), 0);
        }
        return new NumericType(parameters.get(0), parameters.get(1));
    }

    public NumericType(@Nullable Integer precision, @Nullable Integer scale) {
        if (scale != null) {
            if (precision == null) {
                throw new IllegalArgumentException("If scale is set for NUMERIC, precision must be set too");
            }
            if (scale > precision) {
                throw new IllegalArgumentException(String.format(Locale.ENGLISH, "Scale of numeric must be less or equal the precision. NUMERIC(%d, %d) is unsupported.", precision, scale));
            }
            if (scale < 0) {
                throw new IllegalArgumentException("Scale of NUMERIC must not be negative");
            }
        }
        this.precision = precision;
        this.scale = scale;
        this.mathContext = precision == null ? MathContext.UNLIMITED : new MathContext(precision);
    }

    public NumericType(StreamInput in) throws IOException {
        this.precision = in.readOptionalVInt();
        this.scale = in.readOptionalVInt();
        this.mathContext = this.precision == null ? MathContext.UNLIMITED : new MathContext(this.precision);
    }

    @Override
    public int id() {
        return 22;
    }

    @Override
    public DataType.Precedence precedence() {
        return DataType.Precedence.NUMERIC;
    }

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public Streamer<BigDecimal> streamer() {
        return this;
    }

    @Override
    public BigDecimal implicitCast(Object value) throws IllegalArgumentException, ClassCastException {
        int targetNumIntegralDigits;
        BigDecimal bd;
        if (value == null) {
            return null;
        }
        if (value instanceof BigDecimal) {
            BigDecimal bigDecimal = (BigDecimal)value;
            bd = bigDecimal.round(this.mathContext);
        } else if (value instanceof String || value instanceof Float || value instanceof Double) {
            bd = new BigDecimal(value.toString(), this.mathContext);
        } else if (value instanceof Number) {
            Number number = (Number)value;
            bd = new BigDecimal(BigInteger.valueOf(number.longValue()), this.mathContext);
        } else {
            throw new ClassCastException("Cannot cast '" + String.valueOf(value) + "' to " + this.getName());
        }
        if (this.scale == null) {
            return bd;
        }
        int sourceNumIntegralDigits = bd.precision() - bd.scale();
        if (sourceNumIntegralDigits - (targetNumIntegralDigits = this.precision - this.scale) > 0) {
            throw new ClassCastException("Cannot cast '" + String.valueOf(value) + "' to " + String.valueOf(this) + " as it looses precision");
        }
        return bd.setScale((int)this.scale, this.mathContext.getRoundingMode());
    }

    @Override
    public BigDecimal sanitizeValue(Object value) {
        if (value == null) {
            return null;
        }
        if (value instanceof String) {
            String str = (String)value;
            BigDecimal bigDecimal = new BigDecimal(str, this.mathContext);
            return this.scale == null ? bigDecimal : bigDecimal.setScale((int)this.scale, this.mathContext.getRoundingMode());
        }
        if (value instanceof Long) {
            Long longValue = (Long)value;
            BigInteger bigInt = BigInteger.valueOf(longValue);
            return new BigDecimal(bigInt, this.scale == null ? 0 : this.scale, this.mathContext);
        }
        return (BigDecimal)value;
    }

    @Override
    DataType<?> merge(DataType<?> other) {
        if (DataTypes.isNumeric(other)) {
            return INSTANCE;
        }
        return super.merge(other);
    }

    @Override
    public BigDecimal valueForInsert(BigDecimal value) {
        return value;
    }

    @Override
    public ColumnType<Expression> toColumnType(@Nullable Supplier<List<ColumnDefinition<Expression>>> convertChildColumn) {
        if (this.scale == null) {
            if (this.precision == null) {
                return new ColumnType(this.getName());
            }
            return new ColumnType(this.getName(), List.of(this.precision));
        }
        return new ColumnType(this.getName(), List.of(this.precision, this.scale));
    }

    public static long size(@NotNull BigDecimal value) {
        return 36L + (long)value.unscaledValue().bitLength() / 8L + 1L;
    }

    public static long sizeDiff(@NotNull BigDecimal first, @NotNull BigDecimal second) {
        return NumericType.size(first) - NumericType.size(second);
    }

    @Override
    public Integer numericPrecision() {
        return this.precision;
    }

    @Nullable
    public Integer scale() {
        return this.scale;
    }

    public MathContext mathContext() {
        return this.mathContext;
    }

    private boolean unscaled() {
        return this.precision == null;
    }

    @Override
    public TypeSignature getTypeSignature() {
        if (this.unscaled()) {
            return super.getTypeSignature();
        }
        ArrayList<TypeSignature> parameters = new ArrayList<TypeSignature>();
        parameters.add(TypeSignature.of(this.precision));
        if (this.scale != null) {
            parameters.add(TypeSignature.of(this.scale));
        }
        return new TypeSignature(this.getName(), parameters);
    }

    @Override
    public List<DataType<?>> getTypeParameters() {
        if (this.unscaled()) {
            return List.of();
        }
        if (this.scale != null) {
            return List.of(DataTypes.INTEGER);
        }
        return List.of(DataTypes.INTEGER, DataTypes.INTEGER);
    }

    @Override
    public int compare(BigDecimal o1, BigDecimal o2) {
        return o1.compareTo(o2);
    }

    @Override
    public BigDecimal readValueFrom(StreamInput in) throws IOException {
        if (in.readBoolean()) {
            int scale;
            byte[] bytes = in.readByteArray();
            int n = scale = this.scale == null ? 0 : this.scale;
            if (in.getVersion().onOrAfter(Version.V_5_9_0) && in.readBoolean()) {
                scale = in.readVInt();
                assert (this.scale == null || this.scale == scale) : "streamed value scale differs from type scale";
            }
            return new BigDecimal(new BigInteger(bytes), scale, this.mathContext);
        }
        return null;
    }

    @Override
    public void writeValueTo(StreamOutput out, BigDecimal v) throws IOException {
        if (v != null) {
            out.writeBoolean(true);
            out.writeByteArray(v.unscaledValue().toByteArray());
            if (out.getVersion().onOrAfter(Version.V_5_9_0)) {
                if (this.scale == null) {
                    out.writeBoolean(true);
                    out.writeVInt(v.scale());
                } else {
                    out.writeBoolean(false);
                }
            }
        } else {
            out.writeBoolean(false);
        }
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        out.writeOptionalVInt(this.precision);
        out.writeOptionalVInt(this.scale);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        NumericType that = (NumericType)o;
        return Objects.equals(this.scale, that.scale) && Objects.equals(this.precision, that.precision);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), this.scale, this.precision);
    }

    @Override
    public long valueBytes(BigDecimal value) {
        if (value == null) {
            return RamUsageEstimator.NUM_BYTES_OBJECT_HEADER;
        }
        return NumericType.size(value);
    }

    @Override
    @Nullable
    public StorageSupport<? super BigDecimal> storageSupport() {
        return new NumericStorage(this);
    }

    @Override
    public void addMappingOptions(Map<String, Object> mapping) {
        if (this.precision == null || this.scale == null) {
            throw new UnsupportedOperationException("NUMERIC storage is only supported if precision and scale are specified");
        }
        if (this.maxBytes() > 16) {
            throw new UnsupportedOperationException("Precision for NUMERIC(" + this.precision + ") is too large. Only up to 38 can be stored");
        }
        mapping.put("precision", this.precision);
        mapping.put("scale", this.scale);
    }

    @Override
    public String toString() {
        if (this.getTypeParameters().isEmpty()) {
            return NAME;
        }
        if (this.scale == null) {
            return "numeric(" + this.precision + ")";
        }
        return "numeric(" + this.precision + "," + this.scale + ")";
    }

    public int maxBytes() {
        if (this.precision == null) {
            return Integer.MAX_VALUE;
        }
        return new BigInteger("9".repeat(this.precision)).bitLength() / 8 + 1;
    }

    public BigInteger minValue() {
        if (this.precision == null) {
            throw new UnsupportedOperationException("Can't get min value for numeric type without precision");
        }
        return this.maxValue().negate();
    }

    public BigInteger maxValue() {
        if (this.precision == null) {
            throw new UnsupportedOperationException("Can't get max value for numeric type without precision");
        }
        return new BigInteger("9".repeat(this.precision));
    }
}

