/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import io.crate.common.unit.TimeValue;
import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.cluster.coordination.CoordinationStateRejectedException;
import org.elasticsearch.cluster.coordination.NodeHealthCheckFailureException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.monitor.NodeHealthService;
import org.elasticsearch.monitor.StatusInfo;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.NodeDisconnectedException;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportConnectionListener;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;
import org.jetbrains.annotations.Nullable;

public class LeaderChecker {
    private static final Logger LOGGER = LogManager.getLogger(LeaderChecker.class);
    public static final String LEADER_CHECK_ACTION_NAME = "internal:coordination/fault_detection/leader_check";
    public static final Setting<TimeValue> LEADER_CHECK_INTERVAL_SETTING = Setting.timeSetting("cluster.fault_detection.leader_check.interval", TimeValue.timeValueMillis((long)1000L), TimeValue.timeValueMillis((long)100L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> LEADER_CHECK_TIMEOUT_SETTING = Setting.timeSetting("cluster.fault_detection.leader_check.timeout", TimeValue.timeValueMillis((long)10000L), TimeValue.timeValueMillis((long)1L), Setting.Property.NodeScope);
    public static final Setting<Integer> LEADER_CHECK_RETRY_COUNT_SETTING = Setting.intSetting("cluster.fault_detection.leader_check.retry_count", 3, 1, Setting.Property.NodeScope);
    private final TimeValue leaderCheckInterval;
    private final TimeValue leaderCheckTimeout;
    private final int leaderCheckRetryCount;
    private final TransportService transportService;
    private final Consumer<Exception> onLeaderFailure;
    private final NodeHealthService nodeHealthService;
    private AtomicReference<CheckScheduler> currentChecker = new AtomicReference();
    private volatile DiscoveryNodes discoveryNodes;

    LeaderChecker(Settings settings, TransportService transportService, Consumer<Exception> onLeaderFailure, NodeHealthService nodeHealthService) {
        this.leaderCheckInterval = LEADER_CHECK_INTERVAL_SETTING.get(settings);
        this.leaderCheckTimeout = LEADER_CHECK_TIMEOUT_SETTING.get(settings);
        this.leaderCheckRetryCount = LEADER_CHECK_RETRY_COUNT_SETTING.get(settings);
        this.transportService = transportService;
        this.onLeaderFailure = onLeaderFailure;
        this.nodeHealthService = nodeHealthService;
        transportService.registerRequestHandler(LEADER_CHECK_ACTION_NAME, "same", false, false, LeaderCheckRequest::new, (request, channel) -> {
            this.handleLeaderCheck((LeaderCheckRequest)request);
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        });
        transportService.addConnectionListener(new TransportConnectionListener(){

            @Override
            public void onNodeDisconnected(DiscoveryNode node, Transport.Connection connection) {
                LeaderChecker.this.handleDisconnectedNode(node);
            }
        });
    }

    public DiscoveryNode leader() {
        CheckScheduler checkScheduler = this.currentChecker.get();
        return checkScheduler == null ? null : checkScheduler.leader;
    }

    public void updateLeader(@Nullable DiscoveryNode leader) {
        assert (!this.transportService.getLocalNode().equals(leader));
        CheckScheduler checkScheduler = leader != null ? new CheckScheduler(leader) : null;
        CheckScheduler previousChecker = this.currentChecker.getAndSet(checkScheduler);
        if (previousChecker != null) {
            previousChecker.close();
        }
        if (checkScheduler != null) {
            checkScheduler.handleWakeUp();
        }
    }

    public void setCurrentNodes(DiscoveryNodes discoveryNodes) {
        LOGGER.trace("setCurrentNodes: {}", (Object)discoveryNodes);
        this.discoveryNodes = discoveryNodes;
    }

    boolean currentNodeIsMaster() {
        return this.discoveryNodes.isLocalNodeElectedMaster();
    }

    private void handleLeaderCheck(LeaderCheckRequest request) {
        DiscoveryNodes discoveryNodes = this.discoveryNodes;
        assert (discoveryNodes != null);
        StatusInfo statusInfo = this.nodeHealthService.getHealth();
        if (statusInfo.getStatus() == StatusInfo.Status.UNHEALTHY) {
            String message = "rejecting leader check from [" + String.valueOf(request.getSender()) + "] since node is unhealthy [" + statusInfo.getInfo() + "]";
            LOGGER.debug(message);
            throw new NodeHealthCheckFailureException(message, new Object[0]);
        }
        if (!discoveryNodes.isLocalNodeElectedMaster()) {
            LOGGER.debug("rejecting leader check on non-master {}", (Object)request);
            throw new CoordinationStateRejectedException("rejecting leader check from [" + String.valueOf(request.getSender()) + "] sent to a node that is no longer the master", new Object[0]);
        }
        if (!discoveryNodes.nodeExists(request.getSender())) {
            LOGGER.debug("leader check from unknown node: {}", (Object)request);
            throw new CoordinationStateRejectedException("leader check from unknown node", new Object[0]);
        }
        LOGGER.trace("handling {}", (Object)request);
    }

    private void handleDisconnectedNode(DiscoveryNode discoveryNode) {
        CheckScheduler checkScheduler = this.currentChecker.get();
        if (checkScheduler != null) {
            checkScheduler.handleDisconnectedNode(discoveryNode);
        } else {
            LOGGER.trace("disconnect event ignored for {}, no check scheduler", (Object)discoveryNode);
        }
    }

    private class CheckScheduler
    implements Releasable {
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private final AtomicLong failureCountSinceLastSuccess = new AtomicLong();
        private final DiscoveryNode leader;

        CheckScheduler(DiscoveryNode leader) {
            this.leader = leader;
        }

        @Override
        public void close() {
            if (!this.isClosed.compareAndSet(false, true)) {
                LOGGER.trace("already closed, doing nothing");
            } else {
                LOGGER.debug("closed");
            }
        }

        void handleWakeUp() {
            if (this.isClosed.get()) {
                LOGGER.trace("closed check scheduler woken up, doing nothing");
                return;
            }
            LOGGER.trace("checking {} with [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_TIMEOUT_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckTimeout);
            LeaderChecker.this.transportService.sendRequest(this.leader, LeaderChecker.LEADER_CHECK_ACTION_NAME, (TransportRequest)new LeaderCheckRequest(LeaderChecker.this.transportService.getLocalNode()), TransportRequestOptions.builder().withTimeout(LeaderChecker.this.leaderCheckTimeout).withType(TransportRequestOptions.Type.PING).build(), new TransportResponseHandler<TransportResponse.Empty>(){

                @Override
                public TransportResponse.Empty read(StreamInput in) {
                    return TransportResponse.Empty.INSTANCE;
                }

                @Override
                public void handleResponse(TransportResponse.Empty response) {
                    if (CheckScheduler.this.isClosed.get()) {
                        LOGGER.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    CheckScheduler.this.failureCountSinceLastSuccess.set(0L);
                    CheckScheduler.this.scheduleNextWakeUp();
                }

                @Override
                public void handleException(TransportException exp) {
                    if (CheckScheduler.this.isClosed.get()) {
                        LOGGER.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    if (exp instanceof ConnectTransportException || exp.getCause() instanceof ConnectTransportException) {
                        LOGGER.debug((Message)new ParameterizedMessage("leader [{}] disconnected during check", (Object)CheckScheduler.this.leader), (Throwable)exp);
                        CheckScheduler.this.leaderFailed(new ConnectTransportException(CheckScheduler.this.leader, "disconnected during check", exp));
                        return;
                    }
                    if (exp.getCause() instanceof NodeHealthCheckFailureException) {
                        LOGGER.debug((Message)new ParameterizedMessage("leader [{}] health check failed", (Object)CheckScheduler.this.leader), (Throwable)exp);
                        CheckScheduler.this.leaderFailed(new NodeHealthCheckFailureException("node [" + String.valueOf(CheckScheduler.this.leader) + "] failed health checks", new Object[]{exp}));
                        return;
                    }
                    long failureCount = CheckScheduler.this.failureCountSinceLastSuccess.incrementAndGet();
                    if (failureCount >= (long)LeaderChecker.this.leaderCheckRetryCount) {
                        LOGGER.debug((Message)new ParameterizedMessage("leader [{}] has failed {} consecutive checks (limit [{}] is {}); last failure was:", new Object[]{CheckScheduler.this.leader, failureCount, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount}), (Throwable)exp);
                        CheckScheduler.this.leaderFailed(new ElasticsearchException("node [" + String.valueOf(CheckScheduler.this.leader) + "] failed [" + failureCount + "] consecutive checks", (Throwable)exp, new Object[0]));
                        return;
                    }
                    LOGGER.debug((Message)new ParameterizedMessage("{} consecutive failures (limit [{}] is {}) with leader [{}]", new Object[]{failureCount, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount, CheckScheduler.this.leader}), (Throwable)exp);
                    CheckScheduler.this.scheduleNextWakeUp();
                }

                @Override
                public String executor() {
                    return "same";
                }
            });
        }

        void leaderFailed(final Exception e) {
            if (this.isClosed.compareAndSet(false, true)) {
                LeaderChecker.this.transportService.getThreadPool().generic().execute(new Runnable(){
                    final /* synthetic */ CheckScheduler this$1;
                    {
                        this.this$1 = this$1;
                    }

                    @Override
                    public void run() {
                        this.this$1.LeaderChecker.this.onLeaderFailure.accept(e);
                    }

                    public String toString() {
                        return "notification of leader failure: " + e.getMessage();
                    }
                });
            } else {
                LOGGER.trace("already closed, not failing leader");
            }
        }

        void handleDisconnectedNode(DiscoveryNode discoveryNode) {
            if (discoveryNode.equals(this.leader)) {
                LOGGER.debug("leader [{}] disconnected", (Object)this.leader);
                this.leaderFailed(new NodeDisconnectedException(discoveryNode, "disconnected"));
            }
        }

        private void scheduleNextWakeUp() {
            LOGGER.trace("scheduling next check of {} for [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_INTERVAL_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckInterval);
            LeaderChecker.this.transportService.getThreadPool().schedule(new Runnable(){

                @Override
                public void run() {
                    CheckScheduler.this.handleWakeUp();
                }

                public String toString() {
                    return "scheduled check of leader " + String.valueOf(CheckScheduler.this.leader);
                }
            }, LeaderChecker.this.leaderCheckInterval, "same");
        }
    }

    public static class LeaderCheckRequest
    extends TransportRequest {
        private final DiscoveryNode sender;

        public LeaderCheckRequest(DiscoveryNode sender) {
            this.sender = sender;
        }

        public LeaderCheckRequest(StreamInput in) throws IOException {
            super(in);
            this.sender = new DiscoveryNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.sender.writeTo(out);
        }

        public DiscoveryNode getSender() {
            return this.sender;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LeaderCheckRequest that = (LeaderCheckRequest)o;
            return Objects.equals(this.sender, that.sender);
        }

        public int hashCode() {
            return Objects.hash(this.sender);
        }

        public String toString() {
            return "LeaderCheckRequest{sender=" + String.valueOf(this.sender) + "}";
        }
    }
}

