/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.service;

import io.crate.common.collections.Lists;
import io.crate.common.unit.TimeValue;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.Assertions;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.PlainFuture;
import org.elasticsearch.cluster.AckedClusterStateTaskListener;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ClusterStateTaskExecutor;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.coordination.ClusterStatePublisher;
import org.elasticsearch.cluster.coordination.FailedToCommitClusterStateException;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.service.BatchedTask;
import org.elasticsearch.cluster.service.PendingClusterTask;
import org.elasticsearch.cluster.service.SourcePrioritizedRunnable;
import org.elasticsearch.cluster.service.TaskBatcher;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.CountDown;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.common.util.concurrent.PrioritizedEsThreadPoolExecutor;
import org.elasticsearch.node.Node;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.jetbrains.annotations.Nullable;

public class MasterService
extends AbstractLifecycleComponent {
    private static final Logger LOGGER = LogManager.getLogger(MasterService.class);
    public static final Setting<TimeValue> MASTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING = Setting.positiveTimeSetting("cluster.service.slow_master_task_logging_threshold", TimeValue.timeValueSeconds((long)10L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    static final String MASTER_UPDATE_THREAD_NAME = "masterService#updateTask";
    ClusterStatePublisher clusterStatePublisher;
    private final String nodeName;
    private Supplier<ClusterState> clusterStateSupplier;
    private volatile TimeValue slowTaskLoggingThreshold;
    protected final ThreadPool threadPool;
    private volatile PrioritizedEsThreadPoolExecutor threadPoolExecutor;
    private volatile TaskBatcher<Object> taskBatcher;

    public MasterService(Settings settings, ClusterSettings clusterSettings, ThreadPool threadPool) {
        this.nodeName = Objects.requireNonNull(Node.NODE_NAME_SETTING.get(settings));
        this.slowTaskLoggingThreshold = MASTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING.get(settings);
        clusterSettings.addSettingsUpdateConsumer(MASTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING, this::setSlowTaskLoggingThreshold);
        this.threadPool = threadPool;
    }

    private void setSlowTaskLoggingThreshold(TimeValue slowTaskLoggingThreshold) {
        this.slowTaskLoggingThreshold = slowTaskLoggingThreshold;
    }

    public synchronized void setClusterStatePublisher(ClusterStatePublisher publisher) {
        this.clusterStatePublisher = publisher;
    }

    public synchronized void setClusterStateSupplier(Supplier<ClusterState> clusterStateSupplier) {
        this.clusterStateSupplier = clusterStateSupplier;
    }

    @Override
    protected synchronized void doStart() {
        Objects.requireNonNull(this.clusterStatePublisher, "please set a cluster state publisher before starting");
        Objects.requireNonNull(this.clusterStateSupplier, "please set a cluster state supplier before starting");
        this.threadPoolExecutor = this.createThreadPoolExecutor();
        this.taskBatcher = new TaskBatcher(LOGGER, this.threadPool.generic(), this.threadPoolExecutor, taskInputs -> this.runTasks((TaskInputs)taskInputs));
    }

    protected PrioritizedEsThreadPoolExecutor createThreadPoolExecutor() {
        return EsExecutors.newSinglePrioritizing(this.nodeName + "/masterService#updateTask", EsExecutors.daemonThreadFactory(this.nodeName, MASTER_UPDATE_THREAD_NAME), this.threadPool.scheduler());
    }

    @Override
    protected synchronized void doStop() {
        ThreadPool.terminate(this.threadPoolExecutor, 10L, TimeUnit.SECONDS);
    }

    @Override
    protected synchronized void doClose() {
    }

    ClusterState state() {
        return this.clusterStateSupplier.get();
    }

    private static boolean isMasterUpdateThread() {
        return Thread.currentThread().getName().contains(MASTER_UPDATE_THREAD_NAME);
    }

    public static boolean assertNotMasterUpdateThread(String reason) {
        assert (!MasterService.isMasterUpdateThread()) : "Expected current thread [" + String.valueOf(Thread.currentThread()) + "] to not be the master service thread. Reason: [" + reason + "]";
        return true;
    }

    private <T> void runTasks(TaskInputs<T> taskInputs) {
        String summary = taskInputs.summary;
        if (!this.lifecycle.started()) {
            LOGGER.debug("processing [{}]: ignoring, master service not started", (Object)summary);
            return;
        }
        LOGGER.debug("executing cluster state update for [{}]", (Object)summary);
        ClusterState previousClusterState = this.state();
        if (!previousClusterState.nodes().isLocalNodeElectedMaster() && taskInputs.runOnlyWhenMaster()) {
            LOGGER.debug("failing [{}]: local node is no longer master", (Object)summary);
            taskInputs.onNoLongerMaster();
            return;
        }
        long computationStartTime = this.threadPool.relativeTimeInMillis();
        TaskOutputs<T> taskOutputs = MasterService.calculateTaskOutputs(taskInputs, previousClusterState);
        taskOutputs.notifyFailedTasks();
        TimeValue computationTime = this.getTimeSince(computationStartTime);
        this.logExecutionTime(computationTime, "compute cluster state update", summary);
        if (taskOutputs.clusterStateUnchanged()) {
            long notificationStartTime = this.threadPool.relativeTimeInMillis();
            taskOutputs.notifySuccessfulTasksOnUnchangedClusterState();
            TimeValue executionTime = this.getTimeSince(notificationStartTime);
            this.logExecutionTime(executionTime, "notify listeners on unchanged cluster state", summary);
        } else {
            ClusterState newClusterState = taskOutputs.newClusterState;
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("cluster state updated, source [{}]\n{}", (Object)summary, (Object)newClusterState);
            } else {
                LOGGER.debug("cluster state updated, version [{}], source [{}]", (Object)newClusterState.version(), (Object)summary);
            }
            long publicationStartTime = this.threadPool.relativeTimeInMillis();
            try {
                String nodeSummary;
                ClusterChangedEvent clusterChangedEvent = new ClusterChangedEvent(summary, newClusterState, previousClusterState);
                DiscoveryNodes.Delta nodesDelta = clusterChangedEvent.nodesDelta();
                if (nodesDelta.hasChanges() && LOGGER.isInfoEnabled() && (nodeSummary = nodesDelta.shortSummary()).length() > 0) {
                    LOGGER.info("{}, term: {}, version: {}, reason: {}", (Object)summary, (Object)newClusterState.term(), (Object)newClusterState.version(), (Object)nodeSummary);
                }
                LOGGER.debug("publishing cluster state version [{}]", (Object)newClusterState.version());
                this.publish(clusterChangedEvent, taskOutputs, publicationStartTime);
            }
            catch (Exception e) {
                this.handleException(summary, publicationStartTime, newClusterState, e);
            }
        }
    }

    private TimeValue getTimeSince(long startTimeMillis) {
        return TimeValue.timeValueMillis((long)Math.max(0L, this.threadPool.relativeTimeInMillis() - startTimeMillis));
    }

    protected <T> void publish(ClusterChangedEvent clusterChangedEvent, TaskOutputs<T> taskOutputs, long startTimeMillis) {
        PlainFuture<Void> fut = new PlainFuture<Void>(this){

            @Override
            protected boolean blockingAllowed() {
                return MasterService.isMasterUpdateThread() || super.blockingAllowed();
            }
        };
        this.clusterStatePublisher.publish(clusterChangedEvent, (ActionListener<Void>)fut, taskOutputs.createAckListener(this.threadPool, clusterChangedEvent.state()));
        try {
            FutureUtils.get(fut);
            this.onPublicationSuccess(clusterChangedEvent, taskOutputs);
        }
        catch (Exception e) {
            this.onPublicationFailed(clusterChangedEvent, taskOutputs, startTimeMillis, e);
        }
    }

    <T> void onPublicationSuccess(ClusterChangedEvent clusterChangedEvent, TaskOutputs<T> taskOutputs) {
        long notificationStartTime = this.threadPool.relativeTimeInMillis();
        taskOutputs.processedDifferentClusterState(clusterChangedEvent.previousState(), clusterChangedEvent.state());
        try {
            taskOutputs.clusterStatePublished(clusterChangedEvent);
        }
        catch (Exception e) {
            LOGGER.error(() -> new ParameterizedMessage("exception thrown while notifying executor of new cluster state publication [{}]", (Object)clusterChangedEvent.source()), (Throwable)e);
        }
        TimeValue executionTime = this.getTimeSince(notificationStartTime);
        this.logExecutionTime(executionTime, "notify listeners on successful publication of cluster state (version: " + clusterChangedEvent.state().version() + ", uuid: " + clusterChangedEvent.state().stateUUID() + ")", clusterChangedEvent.source());
    }

    <T> void onPublicationFailed(ClusterChangedEvent clusterChangedEvent, TaskOutputs<T> taskOutputs, long startTimeMillis, Exception exception) {
        if (exception instanceof FailedToCommitClusterStateException) {
            long version = clusterChangedEvent.state().version();
            LOGGER.warn(() -> new ParameterizedMessage("failing [{}]: failed to commit cluster state version [{}]", (Object)clusterChangedEvent.source(), (Object)version), (Throwable)exception);
            taskOutputs.publishingFailed((FailedToCommitClusterStateException)exception);
        } else {
            this.handleException(clusterChangedEvent.source(), startTimeMillis, clusterChangedEvent.state(), exception);
        }
    }

    private void handleException(String summary, long startTimeMillis, ClusterState newClusterState, Exception e) {
        TimeValue executionTime = this.getTimeSince(startTimeMillis);
        long version = newClusterState.version();
        String stateUUID = newClusterState.stateUUID();
        String fullState = newClusterState.toString();
        LOGGER.warn((Message)new ParameterizedMessage("took [{}] and then failed to publish updated cluster state (version: {}, uuid: {}) for [{}]:\n{}", new Object[]{executionTime, version, stateUUID, summary, fullState}), (Throwable)e);
    }

    private static <T> TaskOutputs<T> calculateTaskOutputs(TaskInputs<T> taskInputs, ClusterState previousClusterState) {
        ClusterStateTaskExecutor.ClusterTasksResult<T> clusterTasksResult = MasterService.executeTasks(taskInputs, previousClusterState);
        ClusterState newClusterState = MasterService.patchVersions(previousClusterState, clusterTasksResult);
        return new TaskOutputs<T>(taskInputs, previousClusterState, newClusterState, MasterService.getNonFailedTasks(taskInputs, clusterTasksResult), clusterTasksResult.executionResults);
    }

    private static ClusterState patchVersions(ClusterState previousClusterState, ClusterStateTaskExecutor.ClusterTasksResult<?> executionResult) {
        ClusterState newClusterState = executionResult.resultingState;
        if (previousClusterState != newClusterState) {
            ClusterState.Builder builder = ClusterState.builder(newClusterState).incrementVersion();
            if (previousClusterState.routingTable() != newClusterState.routingTable()) {
                builder.routingTable(RoutingTable.builder(newClusterState.routingTable()).version(newClusterState.routingTable().version() + 1L).build());
            }
            if (previousClusterState.metadata() != newClusterState.metadata()) {
                builder.metadata(Metadata.builder(newClusterState.metadata()).version(newClusterState.metadata().version() + 1L));
            }
            newClusterState = builder.build();
        }
        return newClusterState;
    }

    public <T extends ClusterStateTaskConfig & ClusterStateTaskExecutor<T>> void submitStateUpdateTask(String source, T updateTask) {
        this.submitStateUpdateTask(source, updateTask, updateTask, updateTask, (ClusterStateTaskListener)updateTask);
    }

    public <T> void submitStateUpdateTask(String source, T task, ClusterStateTaskConfig config, ClusterStateTaskExecutor<T> executor, ClusterStateTaskListener listener) {
        this.submitStateUpdateTasks(source, Collections.singletonMap(task, listener), config, executor);
    }

    public List<PendingClusterTask> pendingTasks() {
        return Lists.map(this.threadPoolExecutor.getPending(), pending -> {
            assert (pending.task instanceof SourcePrioritizedRunnable) : "thread pool executor should only use SourcePrioritizedRunnable instances but found: " + pending.task.getClass().getName();
            SourcePrioritizedRunnable task = (SourcePrioritizedRunnable)pending.task;
            return new PendingClusterTask(pending.insertionOrder, pending.priority, task.source(), task.getAgeInMillis(), pending.executing);
        });
    }

    public int numberOfPendingTasks() {
        return this.threadPoolExecutor.getNumberOfPendingTasks();
    }

    public TimeValue getMaxTaskWaitTime() {
        return this.threadPoolExecutor.getMaxTaskWaitTime();
    }

    private static SafeClusterStateTaskListener safe(ClusterStateTaskListener listener) {
        if (listener instanceof AckedClusterStateTaskListener) {
            AckedClusterStateTaskListener ackedListener = (AckedClusterStateTaskListener)listener;
            return new SafeAckedClusterStateTaskListener(ackedListener, LOGGER);
        }
        return new SafeClusterStateTaskListener(listener, LOGGER);
    }

    private void logExecutionTime(TimeValue executionTime, String activity, String summary) {
        if (executionTime.millis() > this.slowTaskLoggingThreshold.millis()) {
            LOGGER.warn("took [{}], which is over [{}], to {} for [{}]", (Object)executionTime, (Object)this.slowTaskLoggingThreshold, (Object)activity, (Object)summary);
        } else {
            LOGGER.debug("took [{}] to {} for [{}]", (Object)executionTime, (Object)activity, (Object)summary);
        }
    }

    private static <T> ClusterStateTaskExecutor.ClusterTasksResult<T> executeTasks(TaskInputs<T> taskInputs, ClusterState previousClusterState) {
        ClusterStateTaskExecutor.ClusterTasksResult<Object> clusterTasksResult;
        try {
            List<Object> inputs = taskInputs.updateTasks.stream().map(tBatchedTask -> tBatchedTask.task).toList();
            clusterTasksResult = taskInputs.executor.execute(previousClusterState, inputs);
            if (previousClusterState != clusterTasksResult.resultingState && previousClusterState.nodes().isLocalNodeElectedMaster() && !clusterTasksResult.resultingState.nodes().isLocalNodeElectedMaster()) {
                throw new AssertionError((Object)"update task submitted to MasterService cannot remove master");
            }
        }
        catch (Exception e) {
            LOGGER.trace(() -> new ParameterizedMessage("failed to execute cluster state update (on version: [{}], uuid: [{}]) for [{}]\n{}{}{}", new Object[]{previousClusterState.version(), previousClusterState.stateUUID(), taskInputs.summary, previousClusterState.nodes(), previousClusterState.routingTable(), previousClusterState.getRoutingNodes()}), (Throwable)e);
            clusterTasksResult = ClusterStateTaskExecutor.ClusterTasksResult.builder().failures(taskInputs.updateTasks.stream().map(updateTask -> updateTask.task)::iterator, e).build(previousClusterState);
        }
        assert (clusterTasksResult.executionResults != null);
        assert (clusterTasksResult.executionResults.size() == taskInputs.updateTasks.size()) : String.format(Locale.ROOT, "expected [%d] task result%s but was [%d]", taskInputs.updateTasks.size(), taskInputs.updateTasks.size() == 1 ? "" : "s", clusterTasksResult.executionResults.size());
        if (Assertions.ENABLED) {
            ClusterStateTaskExecutor.ClusterTasksResult<Object> finalClusterTasksResult = clusterTasksResult;
            taskInputs.updateTasks.forEach(updateTask -> {
                assert (finalClusterTasksResult.executionResults.containsKey(updateTask.task)) : "missing task result for " + String.valueOf(updateTask);
            });
        }
        return clusterTasksResult;
    }

    private static <T> List<BatchedTask<T>> getNonFailedTasks(TaskInputs<T> taskInputs, ClusterStateTaskExecutor.ClusterTasksResult<T> clusterTasksResult) {
        return taskInputs.updateTasks.stream().filter(updateTask -> {
            assert (clusterTasksResult.executionResults.containsKey(updateTask.task)) : "missing " + String.valueOf(updateTask);
            ClusterStateTaskExecutor.TaskResult taskResult = clusterTasksResult.executionResults.get(updateTask.task);
            return taskResult.isSuccess();
        }).collect(Collectors.toList());
    }

    public <T> void submitStateUpdateTasks(String source, Map<T, ClusterStateTaskListener> tasks, ClusterStateTaskConfig config, ClusterStateTaskExecutor<T> executor) {
        block3: {
            if (!this.lifecycle.started()) {
                return;
            }
            try {
                TaskBatcher<Object> batcher = this.taskBatcher;
                List safeTasks = tasks.entrySet().stream().map(e -> new BatchedTask(batcher, config.priority(), source, e.getKey(), MasterService.safe((ClusterStateTaskListener)e.getValue()), executor)).toList();
                batcher.submitTasks(safeTasks, config.timeout());
            }
            catch (EsRejectedExecutionException e2) {
                if (this.lifecycle.stoppedOrClosed()) break block3;
                throw e2;
            }
        }
    }

    public record TaskInputs<T>(String summary, List<BatchedTask<T>> updateTasks, ClusterStateTaskExecutor<T> executor) {
        boolean runOnlyWhenMaster() {
            return this.executor.runOnlyOnMaster();
        }

        void onNoLongerMaster() {
            this.updateTasks.forEach(task -> task.listener.onNoLongerMaster(task.source()));
        }
    }

    static class TaskOutputs<T> {
        final TaskInputs<T> taskInputs;
        final ClusterState previousClusterState;
        final ClusterState newClusterState;
        final List<BatchedTask<T>> nonFailedTasks;
        final Map<T, ClusterStateTaskExecutor.TaskResult> executionResults;

        TaskOutputs(TaskInputs<T> taskInputs, ClusterState previousClusterState, ClusterState newClusterState, List<BatchedTask<T>> nonFailedTasks, Map<T, ClusterStateTaskExecutor.TaskResult> executionResults) {
            this.taskInputs = taskInputs;
            this.previousClusterState = previousClusterState;
            this.newClusterState = newClusterState;
            this.nonFailedTasks = nonFailedTasks;
            this.executionResults = executionResults;
        }

        void publishingFailed(FailedToCommitClusterStateException t) {
            this.nonFailedTasks.forEach(task -> task.listener.onFailure(task.source(), t));
        }

        void processedDifferentClusterState(ClusterState previousClusterState, ClusterState newClusterState) {
            this.nonFailedTasks.forEach(task -> task.listener.clusterStateProcessed(task.source(), previousClusterState, newClusterState));
        }

        void clusterStatePublished(ClusterChangedEvent clusterChangedEvent) {
            this.taskInputs.executor.clusterStatePublished(clusterChangedEvent);
        }

        ClusterStatePublisher.AckListener createAckListener(ThreadPool threadPool, ClusterState newClusterState) {
            return new DelegatingAckListener(this.nonFailedTasks.stream().filter(task -> task.listener instanceof AckedClusterStateTaskListener).map(task -> new AckCountDownListener((AckedClusterStateTaskListener)task.listener, newClusterState.version(), newClusterState.nodes(), threadPool)).collect(Collectors.toList()));
        }

        boolean clusterStateUnchanged() {
            return this.previousClusterState == this.newClusterState;
        }

        void notifyFailedTasks() {
            for (BatchedTask updateTask : this.taskInputs.updateTasks) {
                assert (this.executionResults.containsKey(updateTask.task)) : "missing " + String.valueOf(updateTask);
                ClusterStateTaskExecutor.TaskResult taskResult = this.executionResults.get(updateTask.task);
                if (taskResult.isSuccess()) continue;
                updateTask.listener.onFailure(updateTask.source(), taskResult.getFailure());
            }
        }

        void notifySuccessfulTasksOnUnchangedClusterState() {
            this.nonFailedTasks.forEach(task -> {
                if (task.listener instanceof AckedClusterStateTaskListener) {
                    ((AckedClusterStateTaskListener)task.listener).onAllNodesAcked(null);
                }
                task.listener.clusterStateProcessed(task.source(), this.newClusterState, this.newClusterState);
            });
        }
    }

    private static class SafeAckedClusterStateTaskListener
    extends SafeClusterStateTaskListener
    implements AckedClusterStateTaskListener {
        private final AckedClusterStateTaskListener listener;
        private final Logger logger;

        SafeAckedClusterStateTaskListener(AckedClusterStateTaskListener listener, Logger logger) {
            super(listener, logger);
            this.listener = listener;
            this.logger = logger;
        }

        @Override
        public boolean mustAck(DiscoveryNode discoveryNode) {
            return this.listener.mustAck(discoveryNode);
        }

        @Override
        public void onAllNodesAcked(@Nullable Exception e) {
            try {
                this.listener.onAllNodesAcked(e);
            }
            catch (Exception inner) {
                inner.addSuppressed(e);
                this.logger.error("exception thrown by listener while notifying on all nodes acked", (Throwable)inner);
            }
        }

        @Override
        public void onAckTimeout() {
            try {
                this.listener.onAckTimeout();
            }
            catch (Exception e) {
                this.logger.error("exception thrown by listener while notifying on ack timeout", (Throwable)e);
            }
        }

        @Override
        public TimeValue ackTimeout() {
            return this.listener.ackTimeout();
        }
    }

    private static class SafeClusterStateTaskListener
    implements ClusterStateTaskListener {
        private final ClusterStateTaskListener listener;
        private final Logger logger;

        SafeClusterStateTaskListener(ClusterStateTaskListener listener, Logger logger) {
            this.listener = listener;
            this.logger = logger;
        }

        @Override
        public void onFailure(String source, Exception e) {
            try {
                this.listener.onFailure(source, e);
            }
            catch (Exception inner) {
                inner.addSuppressed(e);
                this.logger.error(() -> new ParameterizedMessage("exception thrown by listener notifying of failure from [{}]", (Object)source), (Throwable)inner);
            }
        }

        @Override
        public void onNoLongerMaster(String source) {
            try {
                this.listener.onNoLongerMaster(source);
            }
            catch (Exception e) {
                this.logger.error(() -> new ParameterizedMessage("exception thrown by listener while notifying no longer master from [{}]", (Object)source), (Throwable)e);
            }
        }

        @Override
        public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
            try {
                this.listener.clusterStateProcessed(source, oldState, newState);
            }
            catch (Exception e) {
                this.logger.error(() -> new ParameterizedMessage("exception thrown by listener while notifying of cluster state processed from [{}], old cluster state:\n{}\nnew cluster state:\n{}", new Object[]{source, oldState, newState}), (Throwable)e);
            }
        }
    }

    private static class AckCountDownListener
    implements ClusterStatePublisher.AckListener {
        private static final Logger LOGGER = LogManager.getLogger(AckCountDownListener.class);
        private final AckedClusterStateTaskListener ackedTaskListener;
        private final CountDown countDown;
        private final DiscoveryNode masterNode;
        private final ThreadPool threadPool;
        private final long clusterStateVersion;
        private volatile Scheduler.Cancellable ackTimeoutCallback;
        private Exception lastFailure;

        AckCountDownListener(AckedClusterStateTaskListener ackedTaskListener, long clusterStateVersion, DiscoveryNodes nodes, ThreadPool threadPool) {
            this.ackedTaskListener = ackedTaskListener;
            this.clusterStateVersion = clusterStateVersion;
            this.threadPool = threadPool;
            this.masterNode = nodes.getMasterNode();
            int countDown = 0;
            for (DiscoveryNode node : nodes) {
                if (!node.equals(this.masterNode) && !ackedTaskListener.mustAck(node)) continue;
                ++countDown;
            }
            LOGGER.trace("expecting {} acknowledgements for cluster_state update (version: {})", (Object)countDown, (Object)clusterStateVersion);
            this.countDown = new CountDown(countDown + 1);
        }

        @Override
        public void onCommit(TimeValue commitTime) {
            TimeValue timeLeft;
            TimeValue ackTimeout = this.ackedTaskListener.ackTimeout();
            if (ackTimeout == null) {
                ackTimeout = TimeValue.ZERO;
            }
            if ((timeLeft = TimeValue.timeValueNanos((long)Math.max(0L, ackTimeout.nanos() - commitTime.nanos()))).nanos() == 0L) {
                this.onTimeout();
            } else if (this.countDown.countDown()) {
                this.finish();
            } else {
                this.ackTimeoutCallback = this.threadPool.schedule(this::onTimeout, timeLeft, "generic");
                if (this.countDown.isCountedDown()) {
                    this.ackTimeoutCallback.cancel();
                }
            }
        }

        @Override
        public void onNodeAck(DiscoveryNode node, @Nullable Exception e) {
            if (!node.equals(this.masterNode) && !this.ackedTaskListener.mustAck(node)) {
                return;
            }
            if (e == null) {
                LOGGER.trace("ack received from node [{}], cluster_state update (version: {})", (Object)node, (Object)this.clusterStateVersion);
            } else {
                this.lastFailure = e;
                LOGGER.debug(() -> new ParameterizedMessage("ack received from node [{}], cluster_state update (version: {})", (Object)node, (Object)this.clusterStateVersion), (Throwable)e);
            }
            if (this.countDown.countDown()) {
                this.finish();
            }
        }

        private void finish() {
            LOGGER.trace("all expected nodes acknowledged cluster_state update (version: {})", (Object)this.clusterStateVersion);
            if (this.ackTimeoutCallback != null) {
                this.ackTimeoutCallback.cancel();
            }
            this.ackedTaskListener.onAllNodesAcked(this.lastFailure);
        }

        public void onTimeout() {
            if (this.countDown.fastForward()) {
                LOGGER.trace("timeout waiting for acknowledgement for cluster_state update (version: {})", (Object)this.clusterStateVersion);
                this.ackedTaskListener.onAckTimeout();
            }
        }
    }

    private static class DelegatingAckListener
    implements ClusterStatePublisher.AckListener {
        private final List<ClusterStatePublisher.AckListener> listeners;

        private DelegatingAckListener(List<ClusterStatePublisher.AckListener> listeners) {
            this.listeners = listeners;
        }

        @Override
        public void onCommit(TimeValue commitTime) {
            for (ClusterStatePublisher.AckListener listener : this.listeners) {
                listener.onCommit(commitTime);
            }
        }

        @Override
        public void onNodeAck(DiscoveryNode node, @Nullable Exception e) {
            for (ClusterStatePublisher.AckListener listener : this.listeners) {
                listener.onNodeAck(node, e);
            }
        }
    }
}

