/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.azure;

import com.microsoft.azure.storage.AccessCondition;
import com.microsoft.azure.storage.CloudStorageAccount;
import com.microsoft.azure.storage.OperationContext;
import com.microsoft.azure.storage.RetryExponentialRetry;
import com.microsoft.azure.storage.RetryPolicyFactory;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.BlobListingDetails;
import com.microsoft.azure.storage.blob.BlobProperties;
import com.microsoft.azure.storage.blob.CloudBlob;
import com.microsoft.azure.storage.blob.CloudBlobClient;
import com.microsoft.azure.storage.blob.CloudBlobContainer;
import com.microsoft.azure.storage.blob.CloudBlobDirectory;
import com.microsoft.azure.storage.blob.CloudBlockBlob;
import com.microsoft.azure.storage.blob.DeleteSnapshotsOption;
import com.microsoft.azure.storage.blob.ListBlobItem;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.security.InvalidKeyException;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.common.blobstore.BlobMetadata;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.support.PlainBlobMetadata;
import org.elasticsearch.common.settings.SettingsException;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.repositories.azure.AzureStorageSettings;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;

public class AzureStorageService {
    private static final Logger LOGGER = LogManager.getLogger(AzureStorageService.class);
    public static final ByteSizeValue MIN_CHUNK_SIZE = new ByteSizeValue(1L, ByteSizeUnit.BYTES);
    public static final ByteSizeValue MAX_CHUNK_SIZE = new ByteSizeValue(256L, ByteSizeUnit.MB);
    @VisibleForTesting
    AzureStorageSettings storageSettings;

    AzureStorageService(AzureStorageSettings storageSettings) {
        this.storageSettings = storageSettings;
    }

    public ClientOpCtx client() {
        assert (this.storageSettings != null) : "must be initialized before fetching a new client";
        AzureStorageSettings azureStorageSettings = this.storageSettings;
        try {
            return new ClientOpCtx(this.buildClient(azureStorageSettings), () -> this.buildOperationContext(azureStorageSettings));
        }
        catch (IllegalArgumentException | URISyntaxException | InvalidKeyException e) {
            throw new SettingsException("Invalid azure client settings", (Throwable)e);
        }
    }

    protected CloudBlobClient buildClient(AzureStorageSettings azureStorageSettings) throws InvalidKeyException, URISyntaxException {
        CloudBlobClient client = this.createClient(azureStorageSettings);
        long timeout = azureStorageSettings.getTimeout().millis();
        if (timeout > 0L) {
            if (timeout > Integer.MAX_VALUE) {
                throw new IllegalArgumentException("Timeout [" + String.valueOf(azureStorageSettings.getTimeout()) + "] exceeds 2,147,483,647ms.");
            }
            client.getDefaultRequestOptions().setTimeoutIntervalInMs(Integer.valueOf((int)timeout));
        }
        client.getDefaultRequestOptions().setLocationMode(azureStorageSettings.getLocationMode());
        client.getDefaultRequestOptions().setRetryPolicyFactory((RetryPolicyFactory)new RetryExponentialRetry(30000, azureStorageSettings.getMaxRetries()));
        client.getDefaultRequestOptions().setLocationMode(azureStorageSettings.getLocationMode());
        return client;
    }

    protected CloudBlobClient createClient(AzureStorageSettings azureStorageSettings) throws InvalidKeyException, URISyntaxException {
        String connectionString = azureStorageSettings.getConnectString();
        return CloudStorageAccount.parse((String)connectionString).createCloudBlobClient();
    }

    protected OperationContext buildOperationContext(AzureStorageSettings azureStorageSettings) {
        OperationContext context = new OperationContext();
        context.setProxy(azureStorageSettings.getProxy());
        return context;
    }

    void refreshSettings(AzureStorageSettings clientSettings) {
        this.storageSettings = AzureStorageSettings.copy(clientSettings);
    }

    static String blobNameFromUri(URI uri) {
        String path = uri.getPath();
        String[] splits = path.split("/", 3);
        return splits[2];
    }

    public boolean blobExists(String container, String blob) throws URISyntaxException, StorageException {
        ClientOpCtx clientOpCtx = this.client();
        CloudBlobContainer blobContainer = clientOpCtx.cloudBlobClient.getContainerReference(container);
        CloudBlockBlob azureBlob = blobContainer.getBlockBlobReference(blob);
        return azureBlob.exists(null, null, clientOpCtx.opCtx.get());
    }

    public void deleteBlob(String container, String blob) throws URISyntaxException, StorageException {
        ClientOpCtx clientOpCtx = this.client();
        CloudBlobContainer blobContainer = clientOpCtx.cloudBlobClient.getContainerReference(container);
        LOGGER.trace(() -> new ParameterizedMessage("delete blob for container [{}], blob [{}]", (Object)container, (Object)blob));
        CloudBlockBlob azureBlob = blobContainer.getBlockBlobReference(blob);
        LOGGER.trace(() -> new ParameterizedMessage("container [{}]: blob [{}] found. removing.", (Object)container, (Object)blob));
        azureBlob.delete(DeleteSnapshotsOption.NONE, null, null, clientOpCtx.opCtx.get());
    }

    void deleteBlobDirectory(String container, String path) throws URISyntaxException, StorageException, IOException {
        ClientOpCtx clientOpCtx = this.client();
        CloudBlobContainer blobContainer = clientOpCtx.cloudBlobClient.getContainerReference(container);
        for (ListBlobItem blobItem : blobContainer.listBlobs(path, true)) {
            String blobPath = blobItem.getUri().getPath().substring(1 + container.length() + 1);
            try {
                this.deleteBlob(container, blobPath);
            }
            catch (StorageException | URISyntaxException e) {
                throw new IOException("Deleting directory [" + path + "] failed");
            }
        }
    }

    public InputStream getInputStream(String container, String blob, long position, @Nullable Long length) throws URISyntaxException, StorageException, NoSuchFileException {
        ClientOpCtx clientOpCtx = this.client();
        CloudBlockBlob blockBlobReference = clientOpCtx.cloudBlobClient.getContainerReference(container).getBlockBlobReference(blob);
        LOGGER.trace(() -> new ParameterizedMessage("reading container [{}], blob [{}]", (Object)container, (Object)blob));
        return blockBlobReference.openInputStream(position, length, null, null, clientOpCtx.opCtx.get());
    }

    public Map<String, BlobMetadata> listBlobsByPrefix(String container, String keyPath, String prefix) throws URISyntaxException, StorageException {
        HashMap<String, PlainBlobMetadata> blobsBuilder = new HashMap<String, PlainBlobMetadata>();
        EnumSet<BlobListingDetails> enumBlobListingDetails = EnumSet.of(BlobListingDetails.METADATA);
        ClientOpCtx clientOpCtx = this.client();
        CloudBlobContainer blobContainer = clientOpCtx.cloudBlobClient.getContainerReference(container);
        LOGGER.trace(() -> new ParameterizedMessage("listing container [{}], keyPath [{}], prefix [{}]", new Object[]{container, keyPath, prefix}));
        for (ListBlobItem blobItem : blobContainer.listBlobs(keyPath + (prefix == null ? "" : prefix), false, enumBlobListingDetails, null, clientOpCtx.opCtx.get())) {
            URI uri = blobItem.getUri();
            LOGGER.trace(() -> new ParameterizedMessage("blob url [{}]", (Object)uri));
            String blobPath = uri.getPath().substring(1 + container.length() + 1);
            if (!(blobItem instanceof CloudBlob)) continue;
            CloudBlob cloudBlobItem = (CloudBlob)blobItem;
            BlobProperties properties = cloudBlobItem.getProperties();
            String name = blobPath.substring(keyPath.length());
            LOGGER.trace(() -> new ParameterizedMessage("blob url [{}], name [{}], size [{}]", new Object[]{uri, name, properties.getLength()}));
            blobsBuilder.put(name, new PlainBlobMetadata(name, properties.getLength()));
        }
        return Map.copyOf(blobsBuilder);
    }

    public void writeBlob(String container, String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws URISyntaxException, StorageException, IOException {
        LOGGER.trace(() -> new ParameterizedMessage("writeBlob({}, stream, {})", (Object)blobName, (Object)blobSize));
        ClientOpCtx clientOpCtx = this.client();
        CloudBlobContainer blobContainer = clientOpCtx.cloudBlobClient.getContainerReference(container);
        CloudBlockBlob blob = blobContainer.getBlockBlobReference(blobName);
        try {
            AccessCondition accessCondition = failIfAlreadyExists ? AccessCondition.generateIfNotExistsCondition() : AccessCondition.generateEmptyCondition();
            blob.upload(inputStream, blobSize, accessCondition, null, clientOpCtx.opCtx.get());
        }
        catch (StorageException se) {
            if (failIfAlreadyExists && se.getHttpStatusCode() == 409 && "BlobAlreadyExists".equals(se.getErrorCode())) {
                throw new FileAlreadyExistsException(blobName, null, se.getMessage());
            }
            throw se;
        }
        LOGGER.trace(() -> new ParameterizedMessage("writeBlob({}, stream, {}) - done", (Object)blobName, (Object)blobSize));
    }

    public Set<String> children(String container, BlobPath path) throws URISyntaxException, StorageException {
        HashSet<String> blobsBuilder = new HashSet<String>();
        ClientOpCtx clientOpCtx = this.client();
        CloudBlobContainer blobContainer = clientOpCtx.cloudBlobClient.getContainerReference(container);
        String keyPath = path.buildAsString();
        EnumSet<BlobListingDetails> enumBlobListingDetails = EnumSet.of(BlobListingDetails.METADATA);
        for (ListBlobItem blobItem : blobContainer.listBlobs(keyPath, false, enumBlobListingDetails, null, clientOpCtx.opCtx().get())) {
            if (!(blobItem instanceof CloudBlobDirectory)) continue;
            URI uri = blobItem.getUri();
            LOGGER.trace(() -> new ParameterizedMessage("blob url [{}]", (Object)uri));
            String uriPath = uri.getPath();
            blobsBuilder.add(uriPath.substring(1 + container.length() + 1 + keyPath.length(), uriPath.length() - 1));
        }
        return Set.copyOf(blobsBuilder);
    }

    protected record ClientOpCtx(CloudBlobClient cloudBlobClient, Supplier<OperationContext> opCtx) {
    }
}

