/*
 * Decompiled with CFR 0.152.
 */
package io.crate.replication.logical;

import io.crate.action.FutureActionListener;
import io.crate.common.unit.TimeValue;
import io.crate.exceptions.SQLExceptions;
import io.crate.execution.support.RetryListener;
import io.crate.execution.support.RetryRunnable;
import io.crate.replication.logical.LogicalReplicationSettings;
import io.crate.replication.logical.ShardReplicationService;
import io.crate.replication.logical.action.ReplayChangesAction;
import io.crate.replication.logical.action.ShardChangesAction;
import io.crate.replication.logical.exceptions.InvalidShardEngineException;
import io.crate.replication.logical.seqno.RetentionLeaseHelper;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.bulk.BackoffPolicy;
import org.elasticsearch.action.support.replication.ReplicationResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.index.seqno.SeqNoStats;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.jetbrains.annotations.Nullable;

public class ShardReplicationChangesTracker
implements Closeable {
    private static final Logger LOGGER = LogManager.getLogger(ShardReplicationChangesTracker.class);
    private final String subscriptionName;
    private final ShardId shardId;
    private final LogicalReplicationSettings replicationSettings;
    private final ThreadPool threadPool;
    private final Client localClient;
    private final ShardReplicationService shardReplicationService;
    private final String clusterName;
    private final Deque<SeqNoRange> missingBatches = new ArrayDeque<SeqNoRange>();
    private final AtomicLong observedSeqNoAtLeader;
    private final AtomicLong seqNoAlreadyRequested;
    private volatile Scheduler.Cancellable cancellable;
    private volatile boolean closed = false;

    public ShardReplicationChangesTracker(String subscriptionName, IndexShard indexShard, ThreadPool threadPool, LogicalReplicationSettings replicationSettings, ShardReplicationService shardReplicationService, String clusterName, Client client) {
        this.subscriptionName = subscriptionName;
        this.shardId = indexShard.shardId();
        this.replicationSettings = replicationSettings;
        this.threadPool = threadPool;
        this.localClient = client;
        this.shardReplicationService = shardReplicationService;
        this.clusterName = clusterName;
        SeqNoStats seqNoStats = indexShard.seqNoStats();
        this.observedSeqNoAtLeader = new AtomicLong(seqNoStats.getGlobalCheckpoint());
        this.seqNoAlreadyRequested = new AtomicLong(seqNoStats.getMaxSeqNo());
    }

    public void start() {
        LOGGER.debug("[{}] Spawning the shard changes reader", (Object)this.shardId);
        RetryRunnable retryRunnable = this.newRunnable();
        this.cancellable = retryRunnable;
        retryRunnable.run();
    }

    private RetryRunnable newRunnable() {
        return new RetryRunnable(this.threadPool, "logical_replication", this::pollAndProcessPendingChanges, BackoffPolicy.exponentialBackoff());
    }

    private void pollAndProcessPendingChanges() {
        if (this.closed) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[{}] ShardReplicationChangesTracker closed. Stopping tracking", (Object)this.shardId);
            }
            return;
        }
        SeqNoRange rangeToFetch = this.getNextSeqNoRange();
        if (rangeToFetch == null) {
            this.cancellable = this.threadPool.scheduleUnlessShuttingDown(this.replicationSettings.pollDelay(), "logical_replication", this.newRunnable());
            return;
        }
        long fromSeqNo = rangeToFetch.fromSeqNo();
        long toSeqNo = rangeToFetch.toSeqNo();
        CompletableFuture<Client> futureClient = this.shardReplicationService.getRemoteClusterClient(this.shardId.getIndex(), this.subscriptionName);
        ShardChangesAction.Request getPendingChangesRequest = new ShardChangesAction.Request(this.shardId, fromSeqNo, toSeqNo);
        CompletionStage futurePendingChanges = futureClient.thenCompose(remoteClient -> {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[{}] Getting changes {}-{}", (Object)this.shardId, (Object)fromSeqNo, (Object)toSeqNo);
            }
            return remoteClient.execute(ShardChangesAction.INSTANCE, getPendingChangesRequest);
        });
        CompletionStage futureReplicationResponse = ((CompletableFuture)futurePendingChanges).thenCompose(this::replayChanges);
        ((CompletableFuture)futureReplicationResponse).whenComplete((arg_0, arg_1) -> this.lambda$pollAndProcessPendingChanges$1((CompletableFuture)futurePendingChanges, fromSeqNo, toSeqNo, arg_0, arg_1));
    }

    private CompletableFuture<ReplicationResponse> replayChanges(ShardChangesAction.Response response) {
        List<Translog.Operation> translogOps = response.changes();
        if (translogOps.isEmpty()) {
            return CompletableFuture.completedFuture(new ReplicationResponse());
        }
        ReplayChangesAction.Request replayRequest = new ReplayChangesAction.Request(this.shardId, translogOps, response.maxSeqNoOfUpdatesOrDeletes());
        FutureActionListener listener = new FutureActionListener();
        ReplayChangesRetryListener retryListener = new ReplayChangesRetryListener(this.threadPool.scheduler(), l -> this.localClient.execute(ReplayChangesAction.INSTANCE, replayRequest).whenComplete((BiConsumer)l), listener, BackoffPolicy.exponentialBackoff());
        this.localClient.execute(ReplayChangesAction.INSTANCE, replayRequest).whenComplete((BiConsumer)retryListener);
        return listener.thenApply(resp -> {
            ReplicationResponse.ShardInfo shardInfo = resp.getShardInfo();
            if (shardInfo.getFailed() > 0) {
                for (ReplicationResponse.ShardInfo.Failure failure : shardInfo.getFailures()) {
                    LOGGER.error("[{}] Failed replaying changes. Failure: {}", (Object)this.shardId, (Object)failure);
                }
                throw new RuntimeException("Some changes failed while replaying");
            }
            return resp;
        });
    }

    @Nullable
    private SeqNoRange getNextSeqNoRange() {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[{}] Waiting to get batch. requested: {}, leader: {}", (Object)this.shardId, (Object)this.seqNoAlreadyRequested.get(), (Object)this.observedSeqNoAtLeader.get());
        }
        if (this.seqNoAlreadyRequested.get() > this.observedSeqNoAtLeader.get() && this.missingBatches.isEmpty()) {
            return null;
        }
        if (!this.missingBatches.isEmpty()) {
            SeqNoRange missingBatch = this.missingBatches.removeFirst();
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[{}] Fetching missing batch {}-{}", (Object)this.shardId, (Object)missingBatch.fromSeqNo(), (Object)missingBatch.toSeqNo());
            }
            return missingBatch;
        }
        int batchSize = this.replicationSettings.batchSize();
        long fromSeq = this.seqNoAlreadyRequested.getAndAdd(batchSize) + 1L;
        long toSeq = fromSeq + (long)batchSize - 1L;
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[{}] Fetching the batch {}-{}", (Object)this.shardId, (Object)fromSeq, (Object)toSeq);
        }
        return new SeqNoRange(fromSeq, toSeq);
    }

    private void updateBatchFetched(boolean success, long fromSeqNoRequested, long toSeqNoRequested, long toSeqNoReceived, long seqNoAtLeader) {
        if (this.closed) {
            return;
        }
        if (success) {
            assert (toSeqNoRequested >= toSeqNoReceived) : Thread.currentThread().getName() + " Got more operations in the batch than requested";
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[{}] Updating the batch fetched. {}-{}/{}, seqNoAtLeader:{}", (Object)this.shardId, (Object)fromSeqNoRequested, (Object)toSeqNoReceived, (Object)toSeqNoRequested, (Object)seqNoAtLeader);
            }
            if (toSeqNoRequested > toSeqNoReceived && !this.seqNoAlreadyRequested.compareAndSet(toSeqNoRequested, toSeqNoReceived)) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[{}] Didn't get the complete batch. Adding the missing operations {}-{}", (Object)this.shardId, (Object)(toSeqNoReceived + 1L), (Object)toSeqNoRequested);
                }
                this.missingBatches.add(new SeqNoRange(toSeqNoReceived + 1L, toSeqNoRequested));
            }
            long currentSeqNoAtLeader = this.observedSeqNoAtLeader.getAndUpdate(value -> Math.max(seqNoAtLeader, value));
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[{}] observedSeqNoAtLeader: {}", (Object)this.shardId, (Object)currentSeqNoAtLeader);
            }
        } else if (this.seqNoAlreadyRequested.get() == toSeqNoRequested) {
            this.seqNoAlreadyRequested.set(fromSeqNoRequested - 1L);
        } else {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[{}] Adding batch to missing {}-{}", (Object)this.shardId, (Object)fromSeqNoRequested, (Object)toSeqNoRequested);
            }
            this.missingBatches.add(new SeqNoRange(fromSeqNoRequested, toSeqNoRequested));
        }
        this.renewLeasesThenReschedule(toSeqNoReceived);
    }

    private void renewLeasesThenReschedule(long toSeqNoReceived) {
        this.threadPool.executor("logical_replication").execute(() -> this.shardReplicationService.getRemoteClusterClient(this.shardId.getIndex(), this.subscriptionName).thenAccept(remoteClient -> RetentionLeaseHelper.renewRetentionLease(this.shardId, toSeqNoReceived, this.clusterName, remoteClient, ActionListener.wrap(r -> {
            if (!this.closed) {
                this.cancellable = this.threadPool.scheduleUnlessShuttingDown(this.replicationSettings.pollDelay(), "logical_replication", this.newRunnable());
            }
        }, e -> {
            Throwable t = SQLExceptions.unwrap(e);
            boolean isClosed = this.closed;
            if (!isClosed && SQLExceptions.maybeTemporary(t)) {
                LOGGER.info("[{}] Temporary error during renewal of retention leases for subscription '{}'. Retrying: {}:{}", (Object)this.shardId, (Object)this.subscriptionName, (Object)t.getClass().getSimpleName(), (Object)t.getMessage());
                this.cancellable = this.threadPool.scheduleUnlessShuttingDown(this.replicationSettings.pollDelay(), "logical_replication", () -> this.renewLeasesThenReschedule(toSeqNoReceived));
            } else if (isClosed) {
                LOGGER.debug("Exception renewing retention lease. Stopping tracking (closed=true)");
            } else {
                LOGGER.warn("Exception renewing retention lease. Stopping tracking (closed=false)");
            }
        }))));
    }

    @Override
    public void close() throws IOException {
        this.closed = true;
        Scheduler.Cancellable currentCancellable = this.cancellable;
        if (currentCancellable != null) {
            currentCancellable.cancel();
            this.cancellable = null;
        }
        this.shardReplicationService.getRemoteClusterClient(this.shardId.getIndex(), this.subscriptionName).thenAccept(client -> RetentionLeaseHelper.attemptRetentionLeaseRemoval(this.shardId, this.clusterName, client, ActionListener.wrap(() -> {})));
    }

    private /* synthetic */ void lambda$pollAndProcessPendingChanges$1(CompletableFuture futurePendingChanges, long fromSeqNo, long toSeqNo, ReplicationResponse replicationResp, Throwable e) {
        if (e == null) {
            ShardChangesAction.Response pendingChanges = (ShardChangesAction.Response)futurePendingChanges.join();
            List<Translog.Operation> translogOps = pendingChanges.changes();
            long lastSeqNo = translogOps.isEmpty() ? fromSeqNo - 1L : translogOps.get(translogOps.size() - 1).seqNo();
            this.updateBatchFetched(true, fromSeqNo, toSeqNo, lastSeqNo, pendingChanges.lastSyncedGlobalCheckpoint());
        } else {
            Throwable t = SQLExceptions.unwrap(e);
            if (!this.closed && SQLExceptions.maybeTemporary(t)) {
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("[{}] Temporary error during tracking of upstream shard changes for subscription '{}'. Retrying: {}:{}", (Object)this.shardId, (Object)this.subscriptionName, (Object)t.getClass().getSimpleName(), (Object)t.getMessage());
                }
                this.updateBatchFetched(false, fromSeqNo, toSeqNo, fromSeqNo - 1L, -1L);
            } else if (t instanceof InvalidShardEngineException) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Shard is not accepting replayed changes, engine changed", t);
                }
            } else {
                LOGGER.warn("[{}] Error during tracking of upstream shard changes for subscription '{}'. Tracking stopped: {}", (Object)this.shardId, (Object)this.subscriptionName, (Object)t);
            }
        }
    }

    record SeqNoRange(long fromSeqNo, long toSeqNo) {
    }

    private static class ReplayChangesRetryListener<TResp>
    extends RetryListener<TResp> {
        public ReplayChangesRetryListener(ScheduledExecutorService scheduler, Consumer<ActionListener<TResp>> command, ActionListener<TResp> delegate, Iterable<TimeValue> backOffPolicy) {
            super(scheduler, command, delegate, backOffPolicy);
        }

        @Override
        protected boolean shouldRetry(Throwable throwable) {
            return super.shouldRetry(throwable) || throwable instanceof ClusterBlockException;
        }
    }
}

