/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.decider;

import com.carrotsearch.hppc.cursors.ObjectCursor;
import java.util.List;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.RecoverySource;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.allocation.DiskThresholdSettings;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDecider;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.snapshots.SnapshotShardSizeInfo;

public class DiskThresholdDecider
extends AllocationDecider {
    private static final Logger LOGGER = LogManager.getLogger(DiskThresholdDecider.class);
    public static final String NAME = "disk_threshold";
    private final DiskThresholdSettings diskThresholdSettings;

    public DiskThresholdDecider(Settings settings, ClusterSettings clusterSettings) {
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
    }

    public static long sizeOfRelocatingShards(RoutingNode node, boolean subtractShardsMovingAway, String dataPath, ClusterInfo clusterInfo, Metadata metadata, RoutingTable routingTable) {
        String actualPath;
        ClusterInfo.ReservedSpace reservedSpace = clusterInfo.getReservedSpace(node.nodeId(), dataPath);
        long totalSize = reservedSpace.getTotal();
        List<ShardRouting> initializingShards = node.shardsWithState(ShardRoutingState.INITIALIZING);
        initializingShards.removeIf(shardRouting -> reservedSpace.containsShardId(shardRouting.shardId()));
        for (ShardRouting routing : initializingShards) {
            if (routing.relocatingNodeId() == null || (actualPath = clusterInfo.getDataPath(routing)) != null && !actualPath.equals(dataPath)) continue;
            totalSize += DiskThresholdDecider.getExpectedShardSize(routing, 0L, clusterInfo, null, metadata, routingTable);
        }
        if (subtractShardsMovingAway) {
            for (ShardRouting routing : node.shardsWithState(ShardRoutingState.RELOCATING)) {
                actualPath = clusterInfo.getDataPath(routing);
                if (actualPath == null) {
                    actualPath = clusterInfo.getDataPath(routing.cancelRelocation());
                }
                if (!dataPath.equals(actualPath)) continue;
                totalSize -= DiskThresholdDecider.getExpectedShardSize(routing, 0L, clusterInfo, null, metadata, routingTable);
            }
        }
        return totalSize;
    }

    @Override
    public Decision canAllocate(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        boolean skipLowThresholdChecks;
        ClusterInfo clusterInfo = allocation.clusterInfo();
        ImmutableOpenMap<String, DiskUsage> usages = clusterInfo.getNodeMostAvailableDiskUsages();
        Decision decision = this.earlyTerminate(allocation, usages);
        if (decision != null) {
            return decision;
        }
        double usedDiskThresholdLow = 100.0 - this.diskThresholdSettings.getFreeDiskThresholdLow();
        double usedDiskThresholdHigh = 100.0 - this.diskThresholdSettings.getFreeDiskThresholdHigh();
        DiskUsageWithRelocations usage = this.getDiskUsage(node, allocation, usages, false);
        double usedDiskPercentage = usage.getUsedDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (freeBytes < 0L) {
            long sizeOfRelocatingShards = DiskThresholdDecider.sizeOfRelocatingShards(node, false, usage.getPath(), allocation.clusterInfo(), allocation.metadata(), allocation.routingTable());
            LOGGER.debug("fewer free bytes remaining than the size of all incoming shards: usage {} on node {} including {} bytes of relocations, preventing allocation", (Object)usage, (Object)node.nodeId(), (Object)sizeOfRelocatingShards);
            return allocation.decision(Decision.NO, NAME, "the node has fewer free bytes remaining than the total size of all incoming shards: free space [%sB], relocating shards [%sB]", freeBytes + sizeOfRelocatingShards, sizeOfRelocatingShards);
        }
        ByteSizeValue freeBytesValue = new ByteSizeValue(freeBytes);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("node [{}] has {}% used disk", (Object)node.nodeId(), (Object)usedDiskPercentage);
        }
        boolean bl = skipLowThresholdChecks = shardRouting.primary() && !shardRouting.active() && shardRouting.recoverySource().getType() == RecoverySource.Type.EMPTY_STORE;
        if (freeBytes < this.diskThresholdSettings.getFreeBytesThresholdLow().getBytes()) {
            if (!skipLowThresholdChecks) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("less than the required {} free bytes threshold ({} free) on node {}, preventing allocation", (Object)this.diskThresholdSettings.getFreeBytesThresholdLow(), (Object)freeBytesValue, (Object)node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "the node is above the low watermark cluster setting [%s=%s], having less than the minimum required [%s] free space, actual free: [%s]", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getLowWatermarkRaw(), this.diskThresholdSettings.getFreeBytesThresholdLow(), freeBytesValue);
            }
            if (freeBytes > this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes()) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("less than the required {} free bytes threshold ({} free) on node {}, but allowing allocation because primary has never been allocated", (Object)this.diskThresholdSettings.getFreeBytesThresholdLow(), (Object)freeBytesValue, (Object)node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "the node is above the low watermark, but less than the high watermark, and this primary shard has never been allocated before", new Object[0]);
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("less than the required {} free bytes threshold ({} free) on node {}, preventing allocation even though primary has never been allocated", (Object)this.diskThresholdSettings.getFreeBytesThresholdHigh(), (Object)freeBytesValue, (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "the node is above the high watermark cluster setting [%s=%s], having less than the minimum required [%s] free space, actual free: [%s]", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getHighWatermarkRaw(), this.diskThresholdSettings.getFreeBytesThresholdHigh(), freeBytesValue);
        }
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        if (freeDiskPercentage < this.diskThresholdSettings.getFreeDiskThresholdLow()) {
            if (!skipLowThresholdChecks) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("more than the allowed {} used disk threshold ({} used) on node [{}], preventing allocation", (Object)Strings.format1Decimals(usedDiskThresholdLow, "%"), (Object)Strings.format1Decimals(usedDiskPercentage, "%"), (Object)node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "the node is above the low watermark cluster setting [%s=%s], using more disk space than the maximum allowed [%s%%], actual free: [%s%%]", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getLowWatermarkRaw(), usedDiskThresholdLow, freeDiskPercentage);
            }
            if (freeDiskPercentage > this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("more than the allowed {} used disk threshold ({} used) on node [{}], but allowing allocation because primary has never been allocated", (Object)Strings.format1Decimals(usedDiskThresholdLow, "%"), (Object)Strings.format1Decimals(usedDiskPercentage, "%"), (Object)node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "the node is above the low watermark, but less than the high watermark, and this primary shard has never been allocated before", new Object[0]);
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation even though primary has never been allocated", (Object)Strings.format1Decimals(this.diskThresholdSettings.getFreeDiskThresholdHigh(), "%"), (Object)Strings.format1Decimals(freeDiskPercentage, "%"), (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "the node is above the high watermark cluster setting [%s=%s], using more disk space than the maximum allowed [%s%%], actual free: [%s%%]", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getHighWatermarkRaw(), usedDiskThresholdHigh, freeDiskPercentage);
        }
        long shardSize = DiskThresholdDecider.getExpectedShardSize(shardRouting, 0L, allocation.clusterInfo(), allocation.snapshotShardSizeInfo(), allocation.metadata(), allocation.routingTable());
        assert (shardSize >= 0L) : shardSize;
        double freeSpaceAfterShard = this.freeDiskPercentageAfterShardAssigned(usage, shardSize);
        long freeBytesAfterShard = freeBytes - shardSize;
        if (freeBytesAfterShard < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes()) {
            LOGGER.warn("after allocating [{}] node [{}] would have less than the required threshold of {} free (currently {} free, estimated shard size is {}), preventing allocation", (Object)shardRouting, (Object)node.nodeId(), (Object)this.diskThresholdSettings.getFreeBytesThresholdHigh(), (Object)freeBytesValue, (Object)new ByteSizeValue(shardSize));
            return allocation.decision(Decision.NO, NAME, "allocating the shard to this node will bring the node above the high watermark cluster setting [%s=%s] and cause it to have less than the minimum required [%s] of free space (free: [%s], estimated shard size: [%s])", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getHighWatermarkRaw(), this.diskThresholdSettings.getFreeBytesThresholdHigh(), freeBytesValue, new ByteSizeValue(shardSize));
        }
        if (freeSpaceAfterShard < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
            LOGGER.warn("after allocating [{}] node [{}] would have more than the allowed {} free disk threshold ({} free), preventing allocation", (Object)shardRouting, (Object)node.nodeId(), (Object)Strings.format1Decimals(this.diskThresholdSettings.getFreeDiskThresholdHigh(), "%"), (Object)Strings.format1Decimals(freeSpaceAfterShard, "%"));
            return allocation.decision(Decision.NO, NAME, "allocating the shard to this node will bring the node above the high watermark cluster setting [%s=%s] and cause it to use more disk space than the maximum allowed [%s%%] (free space after shard added: [%s%%])", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getHighWatermarkRaw(), usedDiskThresholdHigh, freeSpaceAfterShard);
        }
        assert (freeBytesAfterShard >= 0L) : freeBytesAfterShard;
        return allocation.decision(Decision.YES, NAME, "enough disk for shard on node, free: [%s], shard size: [%s], free after allocating shard: [%s]", freeBytesValue, new ByteSizeValue(shardSize), new ByteSizeValue(freeBytesAfterShard));
    }

    @Override
    public Decision canRemain(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!shardRouting.currentNodeId().equals(node.nodeId())) {
            throw new IllegalArgumentException("Shard [" + String.valueOf(shardRouting) + "] is not allocated on node: [" + node.nodeId() + "]");
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        ImmutableOpenMap<String, DiskUsage> usages = clusterInfo.getNodeLeastAvailableDiskUsages();
        Decision decision = this.earlyTerminate(allocation, usages);
        if (decision != null) {
            return decision;
        }
        DiskUsageWithRelocations usage = this.getDiskUsage(node, allocation, usages, true);
        String dataPath = clusterInfo.getDataPath(shardRouting);
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("node [{}] has {}% free disk ({} bytes)", (Object)node.nodeId(), (Object)freeDiskPercentage, (Object)freeBytes);
        }
        if (dataPath == null || !usage.getPath().equals(dataPath)) {
            return allocation.decision(Decision.YES, NAME, "this shard is not allocated on the most utilized disk and can remain", new Object[0]);
        }
        if (freeBytes < 0L) {
            long sizeOfRelocatingShards = DiskThresholdDecider.sizeOfRelocatingShards(node, true, usage.getPath(), allocation.clusterInfo(), allocation.metadata(), allocation.routingTable());
            LOGGER.debug("fewer free bytes remaining than the size of all incoming shards: usage {} on node {} including {} bytes of relocations, shard cannot remain", (Object)usage, (Object)node.nodeId(), (Object)sizeOfRelocatingShards);
            return allocation.decision(Decision.NO, NAME, "the shard cannot remain on this node because the node has fewer free bytes remaining than the total size of all incoming shards: free space [%s], relocating shards [%s]", freeBytes + sizeOfRelocatingShards, sizeOfRelocatingShards);
        }
        if (freeBytes < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes()) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, shard cannot remain", (Object)this.diskThresholdSettings.getFreeBytesThresholdHigh(), (Object)freeBytes, (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "the shard cannot remain on this node because it is above the high watermark cluster setting [%s=%s] and there is less than the required [%s] free space on node, actual free: [%s]", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getHighWatermarkRaw(), this.diskThresholdSettings.getFreeBytesThresholdHigh(), new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("less than the required {}% free disk threshold ({}% free) on node {}, shard cannot remain", (Object)this.diskThresholdSettings.getFreeDiskThresholdHigh(), (Object)freeDiskPercentage, (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "the shard cannot remain on this node because it is above the high watermark cluster setting [%s=%s] and there is less than the required [%s%%] free disk on node, actual free: [%s%%]", DiskThresholdSettings.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK_SETTING.getKey(), this.diskThresholdSettings.getHighWatermarkRaw(), this.diskThresholdSettings.getFreeDiskThresholdHigh(), freeDiskPercentage);
        }
        return allocation.decision(Decision.YES, NAME, "there is enough disk on this node for the shard to remain, free: [%s]", new ByteSizeValue(freeBytes));
    }

    private DiskUsageWithRelocations getDiskUsage(RoutingNode node, RoutingAllocation allocation, ImmutableOpenMap<String, DiskUsage> usages, boolean subtractLeavingShards) {
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = this.averageUsage(node, usages);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("unable to determine disk usage for {}, defaulting to average across nodes [{} total] [{} free] [{}% free]", (Object)node.nodeId(), (Object)usage.getTotalBytes(), (Object)usage.getFreeBytes(), (Object)usage.getFreeDiskAsPercentage());
            }
        }
        DiskUsageWithRelocations diskUsageWithRelocations = new DiskUsageWithRelocations(usage, this.diskThresholdSettings.includeRelocations() ? DiskThresholdDecider.sizeOfRelocatingShards(node, subtractLeavingShards, usage.getPath(), allocation.clusterInfo(), allocation.metadata(), allocation.routingTable()) : 0L);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("getDiskUsage(subtractLeavingShards={}) returning {}", (Object)subtractLeavingShards, (Object)diskUsageWithRelocations);
        }
        return diskUsageWithRelocations;
    }

    DiskUsage averageUsage(RoutingNode node, ImmutableOpenMap<String, DiskUsage> usages) {
        if (usages.size() == 0) {
            return new DiskUsage(node.nodeId(), node.node().getName(), "_na_", 0L, 0L);
        }
        long totalBytes = 0L;
        long freeBytes = 0L;
        for (ObjectCursor du : usages.values()) {
            totalBytes += ((DiskUsage)du.value).getTotalBytes();
            freeBytes += ((DiskUsage)du.value).getFreeBytes();
        }
        return new DiskUsage(node.nodeId(), node.node().getName(), "_na_", totalBytes / (long)usages.size(), freeBytes / (long)usages.size());
    }

    double freeDiskPercentageAfterShardAssigned(DiskUsageWithRelocations usage, Long shardSize) {
        shardSize = shardSize == null ? 0L : shardSize;
        DiskUsage newUsage = new DiskUsage(usage.getNodeId(), usage.getNodeName(), usage.getPath(), usage.getTotalBytes(), usage.getFreeBytes() - shardSize);
        return newUsage.getFreeDiskAsPercentage();
    }

    private Decision earlyTerminate(RoutingAllocation allocation, ImmutableOpenMap<String, DiskUsage> usages) {
        if (!this.diskThresholdSettings.isEnabled()) {
            return allocation.decision(Decision.YES, NAME, "the disk threshold decider is disabled", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("cluster info unavailable for disk threshold decider, allowing allocation.");
            }
            return allocation.decision(Decision.YES, NAME, "the cluster info is unavailable", new Object[0]);
        }
        if (usages.isEmpty()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("unable to determine disk usages for disk-aware allocation, allowing allocation");
            }
            return allocation.decision(Decision.YES, NAME, "disk usages are unavailable", new Object[0]);
        }
        return null;
    }

    public static long getExpectedShardSize(ShardRouting shard, long defaultValue, ClusterInfo clusterInfo, SnapshotShardSizeInfo snapshotShardSizeInfo, Metadata metadata, RoutingTable routingTable) {
        IndexMetadata indexMetadata = metadata.getIndexSafe(shard.index());
        if (indexMetadata.getResizeSourceIndex() != null && !shard.active() && shard.recoverySource().getType() == RecoverySource.Type.LOCAL_SHARDS) {
            long targetShardSize = 0L;
            Index mergeSourceIndex = indexMetadata.getResizeSourceIndex();
            IndexMetadata sourceIndexMeta = metadata.index(mergeSourceIndex);
            if (sourceIndexMeta != null) {
                Set<ShardId> shardIds = IndexMetadata.selectRecoverFromShards(shard.id(), sourceIndexMeta, indexMetadata.getNumberOfShards());
                for (IndexShardRoutingTable shardRoutingTable : routingTable.index(mergeSourceIndex.getName())) {
                    if (!shardIds.contains(shardRoutingTable.shardId())) continue;
                    targetShardSize += clusterInfo.getShardSize(shardRoutingTable.primaryShard(), 0L);
                }
            }
            return targetShardSize == 0L ? defaultValue : targetShardSize;
        }
        if (shard.unassigned() && shard.recoverySource().getType() == RecoverySource.Type.SNAPSHOT) {
            return snapshotShardSizeInfo.getShardSize(shard, defaultValue);
        }
        return clusterInfo.getShardSize(shard, defaultValue);
    }

    static class DiskUsageWithRelocations {
        private final DiskUsage diskUsage;
        private final long relocatingShardSize;

        DiskUsageWithRelocations(DiskUsage diskUsage, long relocatingShardSize) {
            this.diskUsage = diskUsage;
            this.relocatingShardSize = relocatingShardSize;
        }

        public String toString() {
            return "DiskUsageWithRelocations{diskUsage=" + String.valueOf(this.diskUsage) + ", relocatingShardSize=" + this.relocatingShardSize + "}";
        }

        double getFreeDiskAsPercentage() {
            if (this.getTotalBytes() == 0L) {
                return 100.0;
            }
            return 100.0 * ((double)this.getFreeBytes() / (double)this.getTotalBytes());
        }

        double getUsedDiskAsPercentage() {
            return 100.0 - this.getFreeDiskAsPercentage();
        }

        long getFreeBytes() {
            try {
                return Math.subtractExact(this.diskUsage.getFreeBytes(), this.relocatingShardSize);
            }
            catch (ArithmeticException e) {
                return Long.MAX_VALUE;
            }
        }

        String getPath() {
            return this.diskUsage.getPath();
        }

        String getNodeId() {
            return this.diskUsage.getNodeId();
        }

        String getNodeName() {
            return this.diskUsage.getNodeName();
        }

        long getTotalBytes() {
            return this.diskUsage.getTotalBytes();
        }
    }
}

