/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.gateway;

import io.crate.common.unit.TimeValue;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Manifest;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.gateway.MetaStateService;
import org.elasticsearch.gateway.PersistedClusterStateService;
import org.elasticsearch.gateway.WriteStateException;
import org.elasticsearch.index.Index;

public class IncrementalClusterStateWriter {
    private static final Logger LOGGER = LogManager.getLogger(IncrementalClusterStateWriter.class);
    private final MetaStateService metaStateService;
    private Manifest previousManifest;
    private ClusterState previousClusterState;
    private final LongSupplier relativeTimeMillisSupplier;
    private boolean incrementalWrite;
    private volatile TimeValue slowWriteLoggingThreshold;

    IncrementalClusterStateWriter(Settings settings, ClusterSettings clusterSettings, MetaStateService metaStateService, Manifest manifest, ClusterState clusterState, LongSupplier relativeTimeMillisSupplier) {
        this.metaStateService = metaStateService;
        this.previousManifest = manifest;
        this.previousClusterState = clusterState;
        this.relativeTimeMillisSupplier = relativeTimeMillisSupplier;
        this.incrementalWrite = false;
        this.slowWriteLoggingThreshold = PersistedClusterStateService.SLOW_WRITE_LOGGING_THRESHOLD.get(settings);
        clusterSettings.addSettingsUpdateConsumer(PersistedClusterStateService.SLOW_WRITE_LOGGING_THRESHOLD, this::setSlowWriteLoggingThreshold);
    }

    private void setSlowWriteLoggingThreshold(TimeValue slowWriteLoggingThreshold) {
        this.slowWriteLoggingThreshold = slowWriteLoggingThreshold;
    }

    void setCurrentTerm(long currentTerm) throws WriteStateException {
        Manifest manifest = new Manifest(currentTerm, this.previousManifest.getClusterStateVersion(), this.previousManifest.getGlobalGeneration(), new HashMap<Index, Long>(this.previousManifest.getIndexGenerations()));
        this.metaStateService.writeManifestAndCleanup("current term changed", manifest);
        this.previousManifest = manifest;
    }

    Manifest getPreviousManifest() {
        return this.previousManifest;
    }

    void setIncrementalWrite(boolean incrementalWrite) {
        this.incrementalWrite = incrementalWrite;
    }

    void updateClusterState(ClusterState newState) throws WriteStateException {
        Metadata newMetadata = newState.metadata();
        long startTimeMillis = this.relativeTimeMillisSupplier.getAsLong();
        AtomicClusterStateWriter writer = new AtomicClusterStateWriter(this.metaStateService, this.previousManifest);
        long globalStateGeneration = this.writeGlobalState(writer, newMetadata);
        Map<Index, Long> indexGenerations = this.writeIndicesMetadata(writer, newState);
        Manifest manifest = new Manifest(this.previousManifest.getCurrentTerm(), newState.version(), globalStateGeneration, indexGenerations);
        this.writeManifest(writer, manifest);
        this.previousManifest = manifest;
        this.previousClusterState = newState;
        long durationMillis = this.relativeTimeMillisSupplier.getAsLong() - startTimeMillis;
        TimeValue finalSlowWriteLoggingThreshold = this.slowWriteLoggingThreshold;
        if (durationMillis >= finalSlowWriteLoggingThreshold.millis()) {
            LOGGER.warn("writing cluster state took [{}ms] which is above the warn threshold of [{}]; wrote metadata for [{}] indices and skipped [{}] unchanged indices", (Object)durationMillis, (Object)finalSlowWriteLoggingThreshold, (Object)writer.getIndicesWritten(), (Object)writer.getIndicesSkipped());
        } else {
            LOGGER.debug("writing cluster state took [{}ms]; wrote metadata for [{}] indices and skipped [{}] unchanged indices", (Object)durationMillis, (Object)writer.getIndicesWritten(), (Object)writer.getIndicesSkipped());
        }
    }

    private void writeManifest(AtomicClusterStateWriter writer, Manifest manifest) throws WriteStateException {
        if (!manifest.equals(this.previousManifest)) {
            writer.writeManifestAndCleanup("changed", manifest);
        }
    }

    private Map<Index, Long> writeIndicesMetadata(AtomicClusterStateWriter writer, ClusterState newState) throws WriteStateException {
        Map<Index, Long> previouslyWrittenIndices = this.previousManifest.getIndexGenerations();
        Set<Index> relevantIndices = IncrementalClusterStateWriter.getRelevantIndices(newState);
        HashMap<Index, Long> newIndices = new HashMap<Index, Long>();
        Metadata previousMetadata = this.incrementalWrite ? this.previousClusterState.metadata() : null;
        List<IndexMetadataAction> actions = IncrementalClusterStateWriter.resolveIndexMetadataActions(previouslyWrittenIndices, relevantIndices, previousMetadata, newState.metadata());
        for (IndexMetadataAction action : actions) {
            long generation = action.execute(writer);
            newIndices.put(action.getIndex(), generation);
        }
        return newIndices;
    }

    private long writeGlobalState(AtomicClusterStateWriter writer, Metadata newMetadata) throws WriteStateException {
        if (!this.incrementalWrite || !Metadata.isGlobalStateEquals(this.previousClusterState.metadata(), newMetadata)) {
            return writer.writeGlobalState("changed", newMetadata);
        }
        return this.previousManifest.getGlobalGeneration();
    }

    static List<IndexMetadataAction> resolveIndexMetadataActions(Map<Index, Long> previouslyWrittenIndices, Set<Index> relevantIndices, Metadata previousMetadata, Metadata newMetadata) {
        ArrayList<IndexMetadataAction> actions = new ArrayList<IndexMetadataAction>();
        for (Index index : relevantIndices) {
            IndexMetadata previousIndexMetadata;
            IndexMetadata newIndexMetadata = newMetadata.getIndexSafe(index);
            IndexMetadata indexMetadata = previousIndexMetadata = previousMetadata == null ? null : previousMetadata.index(index);
            if (!previouslyWrittenIndices.containsKey(index) || previousIndexMetadata == null) {
                actions.add(new WriteNewIndexMetadata(newIndexMetadata));
                continue;
            }
            if (previousIndexMetadata.getVersion() != newIndexMetadata.getVersion()) {
                actions.add(new WriteChangedIndexMetadata(previousIndexMetadata, newIndexMetadata));
                continue;
            }
            actions.add(new KeepPreviousGeneration(index, previouslyWrittenIndices.get(index)));
        }
        return actions;
    }

    static Set<Index> getRelevantIndices(ClusterState state) {
        assert (state.nodes().getLocalNode().isDataNode());
        RoutingNode newRoutingNode = state.getRoutingNodes().node(state.nodes().getLocalNodeId());
        if (newRoutingNode == null) {
            throw new IllegalStateException("cluster state does not contain this node - cannot write index meta state");
        }
        HashSet<Index> indices = new HashSet<Index>();
        for (ShardRouting routing : newRoutingNode) {
            indices.add(routing.index());
        }
        return indices;
    }

    static class AtomicClusterStateWriter {
        private static final String FINISHED_MSG = "AtomicClusterStateWriter is finished";
        private final List<Runnable> commitCleanupActions;
        private final List<Runnable> rollbackCleanupActions;
        private final Manifest previousManifest;
        private final MetaStateService metaStateService;
        private boolean finished;
        private int indicesWritten;
        private int indicesSkipped;

        AtomicClusterStateWriter(MetaStateService metaStateService, Manifest previousManifest) {
            this.metaStateService = metaStateService;
            assert (previousManifest != null);
            this.previousManifest = previousManifest;
            this.commitCleanupActions = new ArrayList<Runnable>();
            this.rollbackCleanupActions = new ArrayList<Runnable>();
            this.finished = false;
        }

        long writeGlobalState(String reason, Metadata metadata) throws WriteStateException {
            assert (!this.finished) : "AtomicClusterStateWriter is finished";
            try {
                this.rollbackCleanupActions.add(() -> this.metaStateService.cleanupGlobalState(this.previousManifest.getGlobalGeneration()));
                long generation = this.metaStateService.writeGlobalState(reason, metadata);
                this.commitCleanupActions.add(() -> this.metaStateService.cleanupGlobalState(generation));
                return generation;
            }
            catch (WriteStateException e) {
                this.rollback();
                throw e;
            }
        }

        long writeIndex(String reason, IndexMetadata metadata) throws WriteStateException {
            assert (!this.finished) : "AtomicClusterStateWriter is finished";
            try {
                Index index = metadata.getIndex();
                Long previousGeneration = this.previousManifest.getIndexGenerations().get(index);
                if (previousGeneration != null) {
                    this.rollbackCleanupActions.add(() -> this.metaStateService.cleanupIndex(index, previousGeneration));
                }
                long generation = this.metaStateService.writeIndex(reason, metadata);
                this.commitCleanupActions.add(() -> this.metaStateService.cleanupIndex(index, generation));
                return generation;
            }
            catch (WriteStateException e) {
                this.rollback();
                throw e;
            }
        }

        void writeManifestAndCleanup(String reason, Manifest manifest) throws WriteStateException {
            assert (!this.finished) : "AtomicClusterStateWriter is finished";
            try {
                this.metaStateService.writeManifestAndCleanup(reason, manifest);
                this.commitCleanupActions.forEach(Runnable::run);
                this.finished = true;
            }
            catch (WriteStateException e) {
                if (!e.isDirty()) {
                    this.rollback();
                }
                throw e;
            }
        }

        void rollback() {
            this.rollbackCleanupActions.forEach(Runnable::run);
            this.finished = true;
        }

        void incrementIndicesWritten() {
            ++this.indicesWritten;
        }

        void incrementIndicesSkipped() {
            ++this.indicesSkipped;
        }

        int getIndicesWritten() {
            return this.indicesWritten;
        }

        int getIndicesSkipped() {
            return this.indicesSkipped;
        }
    }

    static interface IndexMetadataAction {
        public Index getIndex();

        public long execute(AtomicClusterStateWriter var1) throws WriteStateException;
    }

    static class WriteNewIndexMetadata
    implements IndexMetadataAction {
        private final IndexMetadata indexMetadata;

        WriteNewIndexMetadata(IndexMetadata indexMetadata) {
            this.indexMetadata = indexMetadata;
        }

        @Override
        public Index getIndex() {
            return this.indexMetadata.getIndex();
        }

        @Override
        public long execute(AtomicClusterStateWriter writer) throws WriteStateException {
            writer.incrementIndicesWritten();
            return writer.writeIndex("freshly created", this.indexMetadata);
        }
    }

    static class WriteChangedIndexMetadata
    implements IndexMetadataAction {
        private final IndexMetadata newIndexMetadata;
        private final IndexMetadata oldIndexMetadata;

        WriteChangedIndexMetadata(IndexMetadata oldIndexMetadata, IndexMetadata newIndexMetadata) {
            this.oldIndexMetadata = oldIndexMetadata;
            this.newIndexMetadata = newIndexMetadata;
        }

        @Override
        public Index getIndex() {
            return this.newIndexMetadata.getIndex();
        }

        @Override
        public long execute(AtomicClusterStateWriter writer) throws WriteStateException {
            writer.incrementIndicesWritten();
            return writer.writeIndex("version changed from [" + this.oldIndexMetadata.getVersion() + "] to [" + this.newIndexMetadata.getVersion() + "]", this.newIndexMetadata);
        }
    }

    static class KeepPreviousGeneration
    implements IndexMetadataAction {
        private final Index index;
        private final long generation;

        KeepPreviousGeneration(Index index, long generation) {
            this.index = index;
            this.generation = generation;
        }

        @Override
        public Index getIndex() {
            return this.index;
        }

        @Override
        public long execute(AtomicClusterStateWriter writer) {
            writer.incrementIndicesSkipped();
            return this.generation;
        }
    }
}

