/*
 * Decompiled with CFR 0.152.
 */
package io.crate.execution.engine.collect.files;

import io.crate.analyze.CopyStatementSettings;
import io.crate.common.exceptions.Exceptions;
import io.crate.common.unit.TimeValue;
import io.crate.data.BatchIterator;
import io.crate.execution.engine.collect.files.FileInput;
import io.crate.execution.engine.collect.files.FileInputFactory;
import io.crate.execution.engine.collect.files.URLFileInput;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.zip.GZIPInputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.bulk.BackoffPolicy;
import org.elasticsearch.common.settings.Settings;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;

public class FileReadingIterator
implements BatchIterator<LineCursor> {
    private static final Logger LOGGER = LogManager.getLogger(FileReadingIterator.class);
    @VisibleForTesting
    static final int MAX_SOCKET_TIMEOUT_RETRIES = 5;
    private static final Predicate<URI> MATCH_ALL_PREDICATE = input -> true;
    private final Map<String, FileInputFactory> fileInputFactories;
    private final Boolean shared;
    private final int numReaders;
    private final int readerNumber;
    private final boolean compressed;
    private final boolean failFast;
    private final List<FileInput> fileInputs;
    private volatile Throwable killed;
    private Iterator<FileInput> fileInputsIterator = null;
    private FileInput currentInput = null;
    private Iterator<URI> currentInputUriIterator = null;
    private BufferedReader currentReader = null;
    @VisibleForTesting
    long watermark;
    private final LineCursor cursor;
    private final ScheduledExecutorService scheduler;
    private final Iterator<TimeValue> backOffPolicy;

    public FileReadingIterator(Collection<URI> fileUris, String compression, Map<String, FileInputFactory> fileInputFactories, Boolean shared, int numReaders, int readerNumber, Settings withClauseOptions, ScheduledExecutorService scheduler) {
        this.compressed = compression != null && compression.equalsIgnoreCase("gzip");
        this.fileInputFactories = fileInputFactories;
        this.cursor = new LineCursor();
        this.shared = shared;
        this.failFast = CopyStatementSettings.FAIL_FAST_SETTING.get(withClauseOptions);
        this.numReaders = numReaders;
        this.readerNumber = readerNumber;
        this.scheduler = scheduler;
        this.backOffPolicy = BackoffPolicy.exponentialBackoff(TimeValue.ZERO, 5).iterator();
        this.fileInputs = fileUris.stream().map(uri -> this.toFileInput((URI)uri, withClauseOptions)).filter(Objects::nonNull).toList();
        this.fileInputsIterator = this.fileInputs.iterator();
    }

    public LineCursor currentElement() {
        return this.cursor;
    }

    public void kill(@NotNull Throwable throwable) {
        this.killed = throwable;
    }

    public void moveToStart() {
        this.raiseIfKilled();
        this.reset();
        this.watermark = 0L;
        this.fileInputsIterator = this.fileInputs.iterator();
    }

    public boolean moveNext() {
        this.raiseIfKilled();
        try {
            if (this.currentReader != null) {
                String line;
                try {
                    line = this.getLine(this.currentReader);
                }
                catch (SocketException | SocketTimeoutException e) {
                    if (this.backOffPolicy.hasNext()) {
                        return false;
                    }
                    throw e;
                }
                if (line == null) {
                    this.closeReader();
                    return this.moveNext();
                }
                this.cursor.line = line;
                this.cursor.failure = null;
                return true;
            }
            if (this.currentInputUriIterator != null && this.currentInputUriIterator.hasNext()) {
                this.advanceToNextUri(this.currentInput);
                return this.moveNext();
            }
            if (this.fileInputsIterator != null && this.fileInputsIterator.hasNext()) {
                boolean advanced = this.advanceToNextFileInput();
                if (advanced) {
                    return this.moveNext();
                }
                this.closeReader();
                return false;
            }
            this.reset();
            return false;
        }
        catch (IOException e) {
            this.cursor.failure = e;
            this.closeReader();
            if (this.failFast) {
                throw new UncheckedIOException(e);
            }
            if (this.cursor.lineNumber == 0L) {
                return true;
            }
            return this.moveNext();
        }
    }

    private void advanceToNextUri(FileInput fileInput) throws IOException {
        this.watermark = 0L;
        this.createReader(fileInput, this.currentInputUriIterator.next());
    }

    private boolean advanceToNextFileInput() throws IOException {
        this.currentInput = this.fileInputsIterator.next();
        List<URI> allUris = this.currentInput.expandUri();
        List<URI> filteredUris = allUris.stream().filter(this::shouldBeReadByCurrentNode).toList();
        if (!filteredUris.isEmpty()) {
            this.currentInputUriIterator = filteredUris.iterator();
            this.advanceToNextUri(this.currentInput);
            return true;
        }
        if (this.currentInput.isGlobbed()) {
            URI uri;
            this.cursor.uri = uri = this.currentInput.uri();
            if (allUris.isEmpty()) {
                throw new IOException("Cannot find any URI matching: " + uri.toString());
            }
            return false;
        }
        return false;
    }

    private boolean shouldBeReadByCurrentNode(URI uri) {
        boolean sharedStorage = Objects.requireNonNullElse(this.shared, this.currentInput.sharedStorageDefault());
        if (sharedStorage) {
            return FileReadingIterator.moduloPredicateImpl(uri, this.readerNumber, this.numReaders);
        }
        return MATCH_ALL_PREDICATE.test(uri);
    }

    private void createReader(FileInput fileInput, URI uri) throws IOException {
        this.cursor.uri = uri;
        this.cursor.lineNumber = 0L;
        InputStream stream = fileInput.getStream(uri);
        this.currentReader = this.createBufferedReader(stream);
    }

    @VisibleForTesting
    void closeReader() {
        if (this.currentReader != null) {
            try {
                this.currentReader.close();
            }
            catch (IOException e) {
                LOGGER.error("Unable to close reader for " + String.valueOf(this.cursor.uri), (Throwable)e);
            }
            this.currentReader = null;
        }
    }

    private String getLine(BufferedReader reader) throws IOException {
        String line = null;
        try {
            while ((line = reader.readLine()) != null) {
                ++this.cursor.lineNumber;
                if (this.cursor.lineNumber < this.watermark) continue;
                this.watermark = 0L;
                if (line.length() == 0) {
                    continue;
                }
                break;
            }
        }
        catch (SocketException | SocketTimeoutException e) {
            if (this.backOffPolicy.hasNext()) {
                this.watermark = this.watermark == 0L ? this.cursor.lineNumber + 1L : this.watermark;
                this.closeReader();
                this.createReader(this.currentInput, this.cursor.uri);
            } else {
                URI uri = this.currentInput.uri();
                LOGGER.error("Timeout during COPY FROM '" + uri.toString() + "' after 5 retries", (Throwable)e);
            }
            throw e;
        }
        catch (Exception e) {
            URI uri = this.currentInput.uri();
            LOGGER.error("Error during COPY FROM '" + uri.toString() + "'", (Throwable)e);
            Exceptions.rethrowUnchecked((Throwable)e);
        }
        return line;
    }

    public void close() {
        this.closeReader();
        this.reset();
        this.killed = BatchIterator.CLOSED;
    }

    private void reset() {
        this.fileInputsIterator = null;
        this.currentInputUriIterator = null;
        if (this.currentInput != null) {
            this.currentInput.close();
        }
        this.currentInput = null;
        this.cursor.failure = null;
    }

    public CompletableFuture<?> loadNextBatch() throws IOException {
        if (this.backOffPolicy.hasNext()) {
            CompletableFuture cf = new CompletableFuture();
            this.scheduler.schedule(() -> cf.complete(null), this.backOffPolicy.next().millis(), TimeUnit.MILLISECONDS);
            return cf;
        }
        throw new IllegalStateException("All batches already loaded");
    }

    public boolean allLoaded() {
        return !this.backOffPolicy.hasNext();
    }

    public boolean hasLazyResultSet() {
        return true;
    }

    @VisibleForTesting
    public static URI toURI(String fileUri) {
        if (fileUri.startsWith("/")) {
            return Paths.get(fileUri, new String[0]).toUri();
        }
        URI uri = URI.create(fileUri);
        if (uri.getScheme() == null) {
            throw new IllegalArgumentException("relative fileURIs are not allowed");
        }
        if (uri.getScheme().equals("file") && !uri.getSchemeSpecificPart().startsWith("///")) {
            throw new IllegalArgumentException("Invalid fileURI");
        }
        return uri;
    }

    @Nullable
    private FileInput toFileInput(URI uri, Settings withClauseOptions) {
        FileInputFactory fileInputFactory = this.fileInputFactories.get(uri.getScheme());
        if (fileInputFactory != null) {
            try {
                return fileInputFactory.create(uri, withClauseOptions);
            }
            catch (IOException e) {
                return null;
            }
        }
        return new URLFileInput(uri);
    }

    @VisibleForTesting
    BufferedReader createBufferedReader(InputStream inputStream) throws IOException {
        BufferedReader reader = this.compressed ? new BufferedReader(new InputStreamReader((InputStream)new GZIPInputStream(inputStream), StandardCharsets.UTF_8)) : new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
        return reader;
    }

    @VisibleForTesting
    public static boolean moduloPredicateImpl(URI input, int readerNumber, int numReaders) {
        int hash = input.hashCode();
        if (hash == Integer.MIN_VALUE) {
            hash = 0;
        }
        return Math.abs(hash) % numReaders == readerNumber;
    }

    private void raiseIfKilled() {
        if (this.killed != null) {
            Exceptions.rethrowUnchecked((Throwable)this.killed);
        }
    }

    public static class LineCursor {
        private URI uri;
        private long lineNumber;
        private String line;
        private IOException failure;

        public LineCursor() {
        }

        public LineCursor(URI uri, long lineNumber, @Nullable String line, @Nullable IOException failure) {
            this.uri = uri;
            this.lineNumber = lineNumber;
            this.line = line;
            this.failure = failure;
        }

        public URI uri() {
            return this.uri;
        }

        public long lineNumber() {
            return this.lineNumber;
        }

        @Nullable
        public String line() {
            return this.line;
        }

        @Nullable
        public IOException failure() {
            return this.failure;
        }

        @VisibleForTesting
        public LineCursor copy() {
            return new LineCursor(this.uri, this.lineNumber, this.line, this.failure);
        }

        public String toString() {
            return "LineCursor{" + String.valueOf(this.uri) + ":" + this.lineNumber + ":line=" + this.line + ", failure=" + String.valueOf(this.failure) + "}";
        }

        public int hashCode() {
            return Objects.hash(this.uri, this.lineNumber, this.line, this.failure);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            LineCursor other = (LineCursor)obj;
            return Objects.equals(this.uri, other.uri) && this.lineNumber == other.lineNumber && Objects.equals(this.line, other.line) && Objects.equals(this.failure, other.failure);
        }
    }
}

