/*
 * Decompiled with CFR 0.152.
 */
package io.crate.gcs;

import com.google.api.gax.paging.Page;
import com.google.cloud.BatchResult;
import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageBatch;
import com.google.cloud.storage.StorageException;
import io.crate.common.collections.Iterables;
import io.crate.gcs.GCSBlobContainer;
import io.crate.gcs.GCSRetryingInputStream;
import io.crate.gcs.GCSService;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetadata;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.support.PlainBlobMetadata;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;

public class GCSBlobStore
implements BlobStore {
    private static final Logger LOGGER = LogManager.getLogger(GCSBlobStore.class);
    public static final int LARGE_BLOB_THRESHOLD_BYTE_SIZE = Math.toIntExact(new ByteSizeValue(5L, ByteSizeUnit.MB).getBytes());
    private final String bucketName;
    private final GCSService storageService;
    private final RepositoryMetadata metadata;
    private final int bufferSize;

    public GCSBlobStore(String bucketName, GCSService storageService, RepositoryMetadata metadata, int bufferSize) {
        this.bucketName = bucketName;
        this.storageService = storageService;
        this.metadata = metadata;
        this.bufferSize = bufferSize;
    }

    private Storage client() {
        return this.storageService.client(this.metadata);
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new GCSBlobContainer(path, this);
    }

    public void close() throws IOException {
        this.storageService.closeRepositoryClient(this.metadata.name());
    }

    Map<String, BlobMetadata> listBlobs(String path) throws IOException {
        return this.listBlobsByPrefix(path, "");
    }

    Map<String, BlobMetadata> listBlobsByPrefix(String path, String prefix) {
        String pathPrefix = GCSBlobStore.buildKey(path, prefix);
        HashMap<String, BlobMetadata> result = new HashMap<String, BlobMetadata>();
        for (Blob blob : this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.prefix((String)pathPrefix)}).iterateAll()) {
            assert (blob.getName().startsWith(path));
            if (blob.isDirectory()) continue;
            String suffixName = blob.getName().substring(path.length());
            result.put(suffixName, (BlobMetadata)new PlainBlobMetadata(suffixName, blob.getSize().longValue()));
        }
        return result;
    }

    Map<String, BlobContainer> listChildren(BlobPath path) {
        String pathStr = path.buildAsString();
        HashMap<String, BlobContainer> result = new HashMap<String, BlobContainer>();
        for (Blob blob : this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.prefix((String)pathStr)}).iterateAll()) {
            if (!blob.isDirectory()) continue;
            assert (blob.getName().startsWith(pathStr));
            assert (blob.getName().endsWith("/"));
            String suffixName = blob.getName().substring(pathStr.length(), blob.getName().length() - 1);
            if (suffixName.isEmpty()) continue;
            result.put(suffixName, (BlobContainer)new GCSBlobContainer(path.add(suffixName), this));
        }
        return result;
    }

    boolean blobExists(String blobName) {
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        Blob blob = this.client().get(blobId);
        return blob != null;
    }

    InputStream readBlob(String blobName) throws IOException {
        return new GCSRetryingInputStream(this.client(), BlobId.of((String)this.bucketName, (String)blobName));
    }

    InputStream readBlob(String blobName, long position, long length) throws IOException {
        if (position < 0L) {
            throw new IllegalArgumentException("position must be non-negative");
        }
        if (length < 0L) {
            throw new IllegalArgumentException("length must be non-negative");
        }
        if (length == 0L) {
            return new ByteArrayInputStream(new byte[0]);
        }
        return new GCSRetryingInputStream(this.client(), BlobId.of((String)this.bucketName, (String)blobName), position, Math.addExact(position, length - 1L));
    }

    void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        BlobInfo blobInfo = BlobInfo.newBuilder((String)this.bucketName, (String)blobName).build();
        if (blobSize > this.getLargeBlobThresholdInBytes()) {
            this.writeBlobResumable(blobInfo, inputStream, blobSize, failIfAlreadyExists);
        } else {
            this.writeBlobMultipart(blobInfo, inputStream, blobSize, failIfAlreadyExists);
        }
    }

    long getLargeBlobThresholdInBytes() {
        return LARGE_BLOB_THRESHOLD_BYTE_SIZE;
    }

    private void writeBlobResumable(BlobInfo blobInfo, InputStream inputStream, long size, boolean failIfAlreadyExists) throws IOException {
        Storage.BlobWriteOption[] blobWriteOptionArray;
        assert (inputStream.markSupported());
        inputStream.mark(Integer.MAX_VALUE);
        byte[] buffer = new byte[size < (long)this.bufferSize ? Math.toIntExact(size) : this.bufferSize];
        StorageException storageException = null;
        if (failIfAlreadyExists) {
            Storage.BlobWriteOption[] blobWriteOptionArray2 = new Storage.BlobWriteOption[1];
            blobWriteOptionArray = blobWriteOptionArray2;
            blobWriteOptionArray2[0] = Storage.BlobWriteOption.doesNotExist();
        } else {
            blobWriteOptionArray = new Storage.BlobWriteOption[]{};
        }
        Storage.BlobWriteOption[] writeOptions = blobWriteOptionArray;
        for (int retry = 0; retry < 3; ++retry) {
            try (WriteChannel writeChannel = this.client().writer(blobInfo, writeOptions);){
                Streams.copy((InputStream)inputStream, (OutputStream)Channels.newOutputStream((WritableByteChannel)writeChannel), (byte[])buffer);
                return;
            }
            catch (StorageException se) {
                int errorCode = se.getCode();
                if (errorCode != 410) {
                    if (failIfAlreadyExists && errorCode == 412) {
                        throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
                    }
                    if (storageException != null) {
                        se.addSuppressed((Throwable)storageException);
                    }
                    throw se;
                }
                LOGGER.warn(() -> new ParameterizedMessage("Retrying broken resumable upload session for blob {}", (Object)blobInfo), (Throwable)se);
                storageException = se;
                inputStream.reset();
                continue;
            }
        }
        throw storageException;
    }

    private void writeBlobMultipart(BlobInfo blobInfo, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (blobSize <= this.getLargeBlobThresholdInBytes()) : "large blob uploads should use the resumable upload method";
        byte[] buffer = new byte[Math.toIntExact(blobSize)];
        inputStream.readNBytes(buffer, 0, buffer.length);
        try {
            Storage.BlobTargetOption[] blobTargetOptionArray;
            if (failIfAlreadyExists) {
                Storage.BlobTargetOption[] blobTargetOptionArray2 = new Storage.BlobTargetOption[1];
                blobTargetOptionArray = blobTargetOptionArray2;
                blobTargetOptionArray2[0] = Storage.BlobTargetOption.doesNotExist();
            } else {
                blobTargetOptionArray = new Storage.BlobTargetOption[]{};
            }
            Storage.BlobTargetOption[] targetOptions = blobTargetOptionArray;
            this.client().create(blobInfo, buffer, targetOptions);
        }
        catch (StorageException se) {
            if (failIfAlreadyExists && se.getCode() == 412) {
                throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
            }
            throw se;
        }
    }

    void deleteDirectory(String pathStr) throws IOException {
        Page page = this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)pathStr)});
        do {
            this.deleteBlobsIgnoringIfNotExists(Iterables.transform((Iterable)page.getValues(), BlobInfo::getName));
        } while ((page = page.getNextPage()) != null);
    }

    void deleteBlobsIgnoringIfNotExists(Iterable<String> blobNames) throws IOException {
        Iterator<String> blobNamesIt = blobNames.iterator();
        if (!blobNamesIt.hasNext()) {
            return;
        }
        final List failedBlobs = Collections.synchronizedList(new ArrayList());
        try {
            final AtomicReference ioe = new AtomicReference();
            StorageBatch batch = this.client().batch();
            for (String blobName : blobNames) {
                final BlobId blob = BlobId.of((String)this.bucketName, (String)blobName);
                batch.delete(blob, new Storage.BlobSourceOption[0]).notify((BatchResult.Callback)new BatchResult.Callback<Boolean, StorageException>(){

                    public void success(Boolean result) {
                    }

                    public void error(StorageException exception) {
                        if (exception.getCode() != 404) {
                            failedBlobs.add(blob);
                            if (!ioe.compareAndSet(null, exception)) {
                                ((StorageException)ioe.get()).addSuppressed((Throwable)exception);
                            }
                        }
                    }
                });
            }
            batch.submit();
            StorageException exception = (StorageException)((Object)ioe.get());
            if (exception != null) {
                throw exception;
            }
        }
        catch (Exception e) {
            throw new IOException("Exception when deleting blobs [" + String.valueOf(failedBlobs) + "]", e);
        }
        assert (failedBlobs.isEmpty());
    }

    private static String buildKey(String keyPath, String s) {
        assert (s != null);
        return keyPath + s;
    }
}

