/*
 * Decompiled with CFR 0.152.
 */
package io.crate.execution.jobs;

import io.crate.concurrent.CompletionListenable;
import io.crate.exceptions.JobKilledException;
import io.crate.exceptions.SQLExceptions;
import io.crate.exceptions.TaskMissing;
import io.crate.execution.engine.collect.stats.JobsLogs;
import io.crate.execution.jobs.Task;
import io.crate.profile.ProfilingContext;
import io.crate.profile.Timer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;

public class RootTask
implements CompletionListenable<Void> {
    private final UUID jobId;
    private final AtomicInteger numActiveTasks;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final Logger logger;
    private final String coordinatorNodeId;
    private final JobsLogs jobsLogs;
    private final CompletableFuture<Void> finishedFuture = new CompletableFuture();
    private final AtomicBoolean killTasksOngoing = new AtomicBoolean(false);
    private final Collection<String> participatedNodes;
    private final String user;
    private final List<Task> orderedTasks;
    @Nullable
    private final ProfilingContext profiler;
    private final boolean traceEnabled;
    private volatile Throwable failure;
    @Nullable
    private final ConcurrentHashMap<Integer, Timer> taskTimersByPhaseId;
    @Nullable
    private final CompletableFuture<Map<String, Object>> profilingFuture;

    private RootTask(Logger logger, UUID jobId, String user, String coordinatorNodeId, Collection<String> participatingNodes, JobsLogs jobsLogs, List<Task> orderedTasks, @Nullable ProfilingContext profilingContext) throws Exception {
        this.logger = logger;
        this.user = user;
        this.coordinatorNodeId = coordinatorNodeId;
        this.participatedNodes = participatingNodes;
        this.jobId = jobId;
        this.jobsLogs = jobsLogs;
        this.orderedTasks = orderedTasks;
        int numTasks = orderedTasks.size();
        this.numActiveTasks = new AtomicInteger(numTasks);
        this.traceEnabled = logger.isTraceEnabled();
        if (profilingContext == null) {
            this.taskTimersByPhaseId = null;
            this.profilingFuture = null;
            this.profiler = null;
        } else {
            this.profiler = profilingContext;
            this.taskTimersByPhaseId = new ConcurrentHashMap(numTasks);
            this.profilingFuture = new CompletableFuture();
        }
        for (Task task : orderedTasks) {
            jobsLogs.operationStarted(task.id(), jobId, task.name(), task::bytesUsed);
            task.completionFuture().whenComplete((BiConsumer)new TaskFinishedListener(task.id()));
        }
    }

    public UUID jobId() {
        return this.jobId;
    }

    String coordinatorNodeId() {
        return this.coordinatorNodeId;
    }

    Collection<String> participatingNodes() {
        return this.participatedNodes;
    }

    public CompletableFuture<Void> start() {
        if (this.closed.get()) {
            this.logger.trace("job={} killed before start was called", (Object)this.jobId);
            return CompletableFuture.completedFuture(null);
        }
        return this.start(0);
    }

    private CompletableFuture<Void> start(int taskIndex) {
        for (int i = taskIndex; i < this.orderedTasks.size(); ++i) {
            Task task = this.orderedTasks.get(i);
            int phaseId = task.id();
            if (this.profiler != null) {
                String subContextName = ProfilingContext.generateProfilingKey(phaseId, task.name());
                if (this.taskTimersByPhaseId.put(phaseId, this.profiler.createAndStartTimer(subContextName)) != null) {
                    return CompletableFuture.failedFuture(new IllegalArgumentException("Timer for " + phaseId + " already added"));
                }
            }
            try {
                this.logger.trace("Task.start job={} id={} name={}", (Object)this.jobId, (Object)phaseId, (Object)task.name());
                CompletableFuture<Void> started = task.start();
                if (started == null || started.isDone() && !started.isCompletedExceptionally()) continue;
                if (started.isCompletedExceptionally()) {
                    return started;
                }
                return started.thenCompose(void_ -> this.start(taskIndex + 1));
            }
            catch (Throwable t) {
                return CompletableFuture.failedFuture(t);
            }
        }
        Throwable localFailure = this.failure;
        if (localFailure != null) {
            return CompletableFuture.failedFuture(localFailure);
        }
        return CompletableFuture.completedFuture(null);
    }

    @Nullable
    public <T extends Task> T getTaskOrNull(int phaseId) {
        for (Task task : this.orderedTasks) {
            if (task.id() != phaseId) continue;
            return (T)task;
        }
        return null;
    }

    public <T extends Task> T getTask(int phaseId) throws TaskMissing {
        T task = this.getTaskOrNull(phaseId);
        if (task == null) {
            throw new TaskMissing(TaskMissing.Type.CHILD, this.jobId, phaseId);
        }
        return task;
    }

    public long kill(@Nullable String reason) {
        int numKilled = 0;
        if (!this.closed.getAndSet(true)) {
            this.logger.trace("RootTask.kill job={}", (Object)this.jobId);
            if (this.numActiveTasks.get() == 0) {
                this.finish();
            } else {
                for (Task task : this.orderedTasks) {
                    if (task.completionFuture().isDone()) continue;
                    if (this.traceEnabled) {
                        this.logger.trace("Task.kill job={} id={} task={}", (Object)this.jobId, (Object)task.id(), (Object)task);
                    }
                    task.kill(JobKilledException.of(reason));
                    ++numKilled;
                }
            }
        }
        return numKilled;
    }

    private void close() {
        if (this.failure != null) {
            this.finishedFuture.completeExceptionally(this.failure);
        } else {
            this.finishedFuture.complete(null);
        }
    }

    private void finish() {
        if (this.profiler != null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Profiling is enabled. Task will not be closed until results are collected!");
                this.logger.trace("Profiling results for job {}: {}", (Object)this.jobId, this.profiler.getDurationInMSByTimer());
            }
            assert (this.profilingFuture != null) : "profilingFuture must not be null";
            try {
                Map<String, Object> executionTimes = this.executionTimes();
                this.profilingFuture.complete(executionTimes);
            }
            catch (Throwable t) {
                this.profilingFuture.completeExceptionally(t);
            }
            if (this.failure != null) {
                this.finishedFuture.completeExceptionally(this.failure);
            }
        } else {
            this.close();
        }
    }

    public CompletableFuture<Map<String, Object>> finishProfiling() {
        if (this.profiler == null) {
            IllegalStateException stateException = new IllegalStateException(String.format(Locale.ENGLISH, "Tried to finish profiling job [id=%s], but profiling is not enabled.", this.jobId));
            return CompletableFuture.failedFuture(stateException);
        }
        assert (this.profilingFuture != null) : "profilingFuture must not be null";
        return this.profilingFuture.whenComplete((o, t) -> this.close());
    }

    @VisibleForTesting
    Map<String, Object> executionTimes() {
        if (this.profiler == null) {
            return Collections.emptyMap();
        }
        return this.profiler.getDurationInMSByTimer();
    }

    @Override
    public CompletableFuture<Void> completionFuture() {
        return this.finishedFuture;
    }

    public String userName() {
        return this.user;
    }

    public String toString() {
        return "Task{id=" + String.valueOf(this.jobId) + ", tasks=" + String.valueOf(this.orderedTasks) + ", closed=" + String.valueOf(this.closed) + ", participating=" + String.valueOf(this.participatedNodes) + "}";
    }

    private final class TaskFinishedListener
    implements BiConsumer<Void, Throwable> {
        private final int id;

        private TaskFinishedListener(int id) {
            this.id = id;
        }

        private boolean finishIfNeeded() {
            if (RootTask.this.traceEnabled) {
                Object task = RootTask.this.getTask(this.id);
                RootTask.this.logger.trace("Task completed job={} id={} task={} error={}", (Object)RootTask.this.jobId, (Object)this.id, task, (Object)RootTask.this.failure);
            }
            if (RootTask.this.numActiveTasks.decrementAndGet() == 0) {
                RootTask.this.finish();
                return true;
            }
            return false;
        }

        private void onSuccess() {
            RootTask.this.jobsLogs.operationFinished(this.id, RootTask.this.jobId, null);
            this.finishIfNeeded();
        }

        private void onFailure(@NotNull Throwable t) {
            RootTask.this.failure = t = SQLExceptions.unwrap(t);
            RootTask.this.jobsLogs.operationFinished(this.id, RootTask.this.jobId, SQLExceptions.messageOf(t));
            if (this.finishIfNeeded()) {
                return;
            }
            if (RootTask.this.killTasksOngoing.compareAndSet(false, true)) {
                for (Task task : RootTask.this.orderedTasks) {
                    if (task.id() == this.id || task.completionFuture().isDone()) continue;
                    if (RootTask.this.traceEnabled) {
                        RootTask.this.logger.trace("Task id={} failed, killing other task={}", (Object)this.id, (Object)task);
                    }
                    task.kill(t);
                }
            }
        }

        @Override
        public void accept(Void result, Throwable throwable) {
            if (RootTask.this.profiler != null) {
                this.stopTaskTimer();
            }
            if (throwable == null) {
                this.onSuccess();
            } else {
                this.onFailure(throwable);
            }
        }

        private void stopTaskTimer() {
            assert (RootTask.this.profiler != null) : "profiler must not be null";
            assert (RootTask.this.taskTimersByPhaseId != null) : "taskTimersByPhaseId must not be null";
            Timer removed = RootTask.this.taskTimersByPhaseId.remove(this.id);
            assert (removed != null) : "removed must not be null";
            RootTask.this.profiler.stopTimerAndStoreDuration(removed);
        }
    }

    public static class Builder {
        private final Logger logger;
        private final UUID jobId;
        private final String coordinatorNode;
        private final JobsLogs jobsLogs;
        private final List<Task> tasks = new ArrayList<Task>();
        private final String user;
        private final Set<String> participatingNodes;
        @Nullable
        private ProfilingContext profilingContext = null;

        Builder(Logger logger, UUID jobId, String user, String coordinatorNode, Collection<String> participatingNodes, JobsLogs jobsLogs) {
            this.logger = logger;
            this.jobId = jobId;
            this.user = user;
            this.coordinatorNode = coordinatorNode;
            this.participatingNodes = new HashSet<String>(participatingNodes);
            this.jobsLogs = jobsLogs;
        }

        public Builder profilingContext(ProfilingContext profilingContext) {
            this.profilingContext = profilingContext;
            return this;
        }

        public void addTask(Task task) {
            assert (this.tasks.stream().noneMatch(x -> x.id() == task.id())) : "Task with id=" + task.id() + " already registered. " + String.valueOf(this.tasks);
            this.tasks.add(task);
        }

        public void addParticipants(Collection<String> participants) {
            this.participatingNodes.addAll(participants);
        }

        boolean isEmpty() {
            return this.tasks.isEmpty();
        }

        int size() {
            return this.tasks.size();
        }

        public UUID jobId() {
            return this.jobId;
        }

        RootTask build() throws Exception {
            return new RootTask(this.logger, this.jobId, this.user, this.coordinatorNode, this.participatingNodes, this.jobsLogs, this.tasks, this.profilingContext);
        }
    }
}

