/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation;

import com.carrotsearch.hppc.ObjectLookupContainer;
import com.carrotsearch.hppc.cursors.ObjectObjectCursor;
import io.crate.common.collections.Sets;
import io.crate.concurrent.MultiActionListener;
import io.crate.metadata.IndexName;
import io.crate.metadata.PartitionName;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.settings.put.UpdateSettingsRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.routing.RerouteService;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.allocation.DiskThresholdSettings;
import org.elasticsearch.cluster.routing.allocation.decider.DiskThresholdDecider;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;

public class DiskThresholdMonitor {
    private static final Logger LOGGER = LogManager.getLogger(DiskThresholdMonitor.class);
    private final DiskThresholdSettings diskThresholdSettings;
    private final Client client;
    private final Supplier<ClusterState> clusterStateSupplier;
    private final LongSupplier currentTimeMillisSupplier;
    private final Supplier<RerouteService> rerouteService;
    private final AtomicLong lastRunTimeMillis = new AtomicLong(Long.MIN_VALUE);
    private final AtomicBoolean checkInProgress = new AtomicBoolean();
    private final Set<String> nodesOverLowThreshold = Sets.newConcurrentHashSet();
    private final Set<String> nodesOverHighThreshold = Sets.newConcurrentHashSet();
    private final Set<String> nodesOverHighThresholdAndRelocating = Sets.newConcurrentHashSet();

    public DiskThresholdMonitor(Settings settings, Supplier<ClusterState> clusterStateSupplier, ClusterSettings clusterSettings, Client client, LongSupplier currentTimeMillisSupplier, Supplier<RerouteService> rerouteService) {
        this.clusterStateSupplier = clusterStateSupplier;
        this.currentTimeMillisSupplier = currentTimeMillisSupplier;
        this.rerouteService = rerouteService;
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
        this.client = client;
    }

    private void checkFinished() {
        boolean checkFinished = this.checkInProgress.compareAndSet(true, false);
        assert (checkFinished);
        LOGGER.trace("checkFinished");
    }

    public void onNewInfo(ClusterInfo info) {
        if (!this.checkInProgress.compareAndSet(false, true)) {
            LOGGER.info("skipping monitor as a check is already in progress");
            return;
        }
        ImmutableOpenMap<String, DiskUsage> usages = info.getNodeLeastAvailableDiskUsages();
        if (usages == null) {
            LOGGER.trace("skipping monitor as no disk usage information is available");
            this.checkFinished();
            return;
        }
        LOGGER.trace("processing new cluster info");
        boolean reroute = false;
        String explanation = "";
        long currentTimeMillis = this.currentTimeMillisSupplier.getAsLong();
        ObjectLookupContainer<String> nodes = usages.keys();
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverLowThreshold);
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverHighThreshold);
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverHighThresholdAndRelocating);
        ClusterState state = this.clusterStateSupplier.get();
        HashSet<String> indicesToMarkReadOnly = new HashSet<String>();
        RoutingNodes routingNodes = state.getRoutingNodes();
        HashSet<String> indicesNotToAutoRelease = new HashSet<String>();
        this.markNodesMissingUsageIneligibleForRelease(routingNodes, usages, indicesNotToAutoRelease);
        ArrayList<DiskUsage> usagesOverHighThreshold = new ArrayList<DiskUsage>();
        for (ObjectObjectCursor<String, DiskUsage> objectObjectCursor : usages) {
            String indexName;
            String node = (String)objectObjectCursor.key;
            DiskUsage usage = (DiskUsage)objectObjectCursor.value;
            RoutingNode routingNode = routingNodes.node(node);
            if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdFloodStage().getBytes() || usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdFloodStage()) {
                this.nodesOverLowThreshold.add(node);
                this.nodesOverHighThreshold.add(node);
                this.nodesOverHighThresholdAndRelocating.remove(node);
                if (routingNode != null) {
                    for (ShardRouting routing : routingNode) {
                        indexName = routing.index().getName();
                        indicesToMarkReadOnly.add(indexName);
                        indicesNotToAutoRelease.add(indexName);
                    }
                }
                LOGGER.warn("flood stage disk watermark [{}] exceeded on {}, all indices on this node will be marked read-only", (Object)this.diskThresholdSettings.describeFloodStageThreshold(), (Object)usage);
                continue;
            }
            if ((usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes() || usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) && routingNode != null) {
                for (ShardRouting routing : routingNode) {
                    indexName = routing.index().getName();
                    indicesNotToAutoRelease.add(indexName);
                }
            }
            long reservedSpace = info.getReservedSpace(usage.getNodeId(), usage.getPath()).getTotal();
            DiskUsage usageWithReservedSpace = new DiskUsage(usage.getNodeId(), usage.getNodeName(), usage.getPath(), usage.getTotalBytes(), Math.max(0L, usage.getFreeBytes() - reservedSpace));
            if (usageWithReservedSpace.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes() || usageWithReservedSpace.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
                this.nodesOverLowThreshold.add(node);
                this.nodesOverHighThreshold.add(node);
                if (this.lastRunTimeMillis.get() <= currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                    reroute = true;
                    explanation = "high disk watermark exceeded on one or more nodes";
                    usagesOverHighThreshold.add(usage);
                    continue;
                }
                LOGGER.debug("high disk watermark exceeded on {} but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
                continue;
            }
            if (usageWithReservedSpace.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdLow().getBytes() || usageWithReservedSpace.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdLow()) {
                this.nodesOverHighThresholdAndRelocating.remove(node);
                boolean wasUnderLowThreshold = this.nodesOverLowThreshold.add(node);
                boolean wasOverHighThreshold = this.nodesOverHighThreshold.remove(node);
                assert (!(wasUnderLowThreshold && wasOverHighThreshold));
                if (wasUnderLowThreshold) {
                    LOGGER.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", (Object)this.diskThresholdSettings.describeLowThreshold(), (Object)usage);
                    continue;
                }
                if (!wasOverHighThreshold) continue;
                LOGGER.info("high disk watermark [{}] no longer exceeded on {}, but low disk watermark [{}] is still exceeded", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)usage, (Object)this.diskThresholdSettings.describeLowThreshold());
                continue;
            }
            this.nodesOverHighThresholdAndRelocating.remove(node);
            if (!this.nodesOverLowThreshold.contains(node)) continue;
            if (this.lastRunTimeMillis.get() <= currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                reroute = true;
                explanation = "one or more nodes has gone under the high or low watermark";
                this.nodesOverLowThreshold.remove(node);
                this.nodesOverHighThreshold.remove(node);
                LOGGER.info("low disk watermark [{}] no longer exceeded on {}", (Object)this.diskThresholdSettings.describeLowThreshold(), (Object)usage);
                continue;
            }
            LOGGER.debug("{} has gone below a disk threshold, but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
        }
        ActionListener<Void> listener = MultiActionListener.of(3, ActionListener.wrap(this::checkFinished));
        if (reroute) {
            LOGGER.debug("rerouting shards: [{}]", (Object)explanation);
            this.rerouteService.get().reroute("disk threshold monitor", Priority.HIGH, ActionListener.wrap(reroutedClusterState -> {
                for (DiskUsage diskUsage : usagesOverHighThreshold) {
                    DiskUsage usageIncludingRelocations;
                    long relocatingShardsSize;
                    RoutingNode routingNode = reroutedClusterState.getRoutingNodes().node(diskUsage.getNodeId());
                    if (routingNode != null) {
                        relocatingShardsSize = this.sizeOfRelocatingShards(routingNode, diskUsage, info, (ClusterState)reroutedClusterState);
                        usageIncludingRelocations = new DiskUsage(diskUsage.getNodeId(), diskUsage.getNodeName(), diskUsage.getPath(), diskUsage.getTotalBytes(), diskUsage.getFreeBytes() - relocatingShardsSize);
                    } else {
                        usageIncludingRelocations = diskUsage;
                        relocatingShardsSize = 0L;
                    }
                    if (usageIncludingRelocations.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes() || usageIncludingRelocations.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
                        this.nodesOverHighThresholdAndRelocating.remove(diskUsage.getNodeId());
                        LOGGER.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes; the node is expected to continue to exceed the high disk watermark when these relocations are complete", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)diskUsage, (Object)(-relocatingShardsSize));
                        continue;
                    }
                    if (this.nodesOverHighThresholdAndRelocating.add(diskUsage.getNodeId())) {
                        LOGGER.info("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes; the node is expected to be below the high disk watermark when these relocations are complete", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)diskUsage, (Object)(-relocatingShardsSize));
                        continue;
                    }
                    LOGGER.debug("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)diskUsage, (Object)(-relocatingShardsSize));
                }
                this.setLastRunTimeMillis();
                listener.onResponse(null);
            }, e -> {
                LOGGER.debug("reroute failed", (Throwable)e);
                this.setLastRunTimeMillis();
                listener.onFailure((Exception)e);
            }));
        } else {
            LOGGER.trace("no reroute required");
            listener.onResponse(null);
        }
        Set<String> set = StreamSupport.stream(state.routingTable().indicesRouting().spliterator(), false).map(c -> (String)c.key).filter(index -> !indicesNotToAutoRelease.contains(index)).filter(index -> state.blocks().hasIndexBlock((String)index, IndexMetadata.INDEX_READ_ONLY_ALLOW_DELETE_BLOCK)).collect(Collectors.toSet());
        if (!set.isEmpty()) {
            LOGGER.info("releasing read-only-allow-delete block on indices: [{}]", set);
            this.updateIndicesReadOnly(set, listener, false);
        } else {
            LOGGER.trace("no auto-release required");
            listener.onResponse(null);
        }
        indicesToMarkReadOnly.removeIf(index -> state.blocks().indexBlocked(ClusterBlockLevel.WRITE, (String)index));
        LOGGER.trace("marking indices as read-only: [{}]", indicesToMarkReadOnly);
        if (!indicesToMarkReadOnly.isEmpty()) {
            this.updateIndicesReadOnly(indicesToMarkReadOnly, listener, true);
        } else {
            listener.onResponse(null);
        }
    }

    long sizeOfRelocatingShards(RoutingNode routingNode, DiskUsage diskUsage, ClusterInfo info, ClusterState reroutedClusterState) {
        return DiskThresholdDecider.sizeOfRelocatingShards(routingNode, true, diskUsage.getPath(), info, reroutedClusterState.metadata(), reroutedClusterState.routingTable());
    }

    private void markNodesMissingUsageIneligibleForRelease(RoutingNodes routingNodes, ImmutableOpenMap<String, DiskUsage> usages, Set<String> indicesToMarkIneligibleForAutoRelease) {
        for (RoutingNode routingNode : routingNodes) {
            if (usages.containsKey(routingNode.nodeId()) || routingNode == null) continue;
            for (ShardRouting routing : routingNode) {
                String indexName = routing.index().getName();
                indicesToMarkIneligibleForAutoRelease.add(indexName);
            }
        }
    }

    private void setLastRunTimeMillis() {
        this.lastRunTimeMillis.getAndUpdate(l -> Math.max(l, this.currentTimeMillisSupplier.getAsLong()));
    }

    protected void updateIndicesReadOnly(Set<String> indicesToUpdate, ActionListener<Void> listener, boolean readOnly) {
        ActionListener<Void> wrappedListener = ActionListener.wrap(r -> {
            this.setLastRunTimeMillis();
            listener.onResponse((Void)r);
        }, e -> {
            LOGGER.debug((Message)new ParameterizedMessage("setting indices [{}] read-only failed", (Object)readOnly), (Throwable)e);
            this.setLastRunTimeMillis();
            listener.onFailure((Exception)e);
        });
        Settings readOnlySettings = readOnly ? Settings.builder().put("index.blocks.read_only_allow_delete", Boolean.TRUE.toString()).build() : Settings.builder().putNull("index.blocks.read_only_allow_delete").build();
        List<PartitionName> partitions = indicesToUpdate.stream().map(index -> IndexName.decode(index).toPartitionName()).toList();
        ((CompletableFuture)this.client.updateSettings(new UpdateSettingsRequest(readOnlySettings, partitions)).thenApply(response -> null)).whenComplete(wrappedListener);
    }

    private static void cleanUpRemovedNodes(ObjectLookupContainer<String> nodesToKeep, Set<String> nodesToCleanUp) {
        for (String node : nodesToCleanUp) {
            if (nodesToKeep.contains((Object)node)) continue;
            nodesToCleanUp.remove(node);
        }
    }
}

