/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.plugins;

import io.crate.common.collections.Tuple;
import io.crate.types.DataTypes;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.analysis.CharFilterFactory;
import org.apache.lucene.analysis.TokenFilterFactory;
import org.apache.lucene.analysis.TokenizerFactory;
import org.apache.lucene.codecs.Codec;
import org.apache.lucene.codecs.DocValuesFormat;
import org.apache.lucene.codecs.PostingsFormat;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.bootstrap.JarHell;
import org.elasticsearch.common.component.LifecycleComponent;
import org.elasticsearch.common.inject.Module;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.IndexModule;
import org.elasticsearch.plugins.Plugin;
import org.elasticsearch.plugins.PluginInfo;

public class PluginsService {
    private static final Logger LOGGER = LogManager.getLogger(PluginsService.class);
    private final Settings settings;
    private final Path configPath;
    private final List<Tuple<PluginInfo, Plugin>> plugins;
    private final Set<Bundle> seenBundles;
    public static final Setting<List<String>> MANDATORY_SETTING = Setting.listSetting("plugin.mandatory", Collections.emptyList(), Function.identity(), DataTypes.STRING_ARRAY, Setting.Property.NodeScope);

    public List<Setting<?>> getPluginSettings() {
        return this.plugins.stream().flatMap(p -> ((Plugin)p.v2()).getSettings().stream()).collect(Collectors.toList());
    }

    public PluginsService(Settings settings, Path configPath, Path pluginsDirectory, Collection<Class<? extends Plugin>> classpathPlugins) {
        this.settings = settings;
        this.configPath = configPath;
        ArrayList<Object> pluginsLoaded = new ArrayList<Object>();
        ArrayList<PluginInfo> pluginsList = new ArrayList<PluginInfo>();
        ArrayList<String> pluginsNames = new ArrayList<String>();
        for (Class<? extends Plugin> clazz : classpathPlugins) {
            Plugin plugin = this.loadPlugin(clazz, settings, configPath);
            PluginInfo pluginInfo = new PluginInfo(clazz.getName(), "classpath plugin", clazz.getName(), Collections.emptyList());
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("plugin loaded from classpath [{}]", (Object)pluginInfo);
            }
            pluginsLoaded.add(new Tuple((Object)pluginInfo, (Object)plugin));
            pluginsList.add(pluginInfo);
            pluginsNames.add(pluginInfo.getName());
        }
        this.seenBundles = new LinkedHashSet<Bundle>();
        if (pluginsDirectory != null) {
            try {
                if (FileSystemUtils.isAccessibleDirectory(pluginsDirectory, LOGGER)) {
                    PluginsService.checkForFailedPluginRemovals(pluginsDirectory);
                    Set<Bundle> plugins = PluginsService.getPluginBundles(pluginsDirectory);
                    for (Bundle bundle : plugins) {
                        pluginsList.add(bundle.plugin);
                        pluginsNames.add(bundle.plugin.getName());
                    }
                    this.seenBundles.addAll(plugins);
                }
            }
            catch (IOException ex) {
                throw new IllegalStateException("Unable to initialize plugins", ex);
            }
        }
        List<Tuple<PluginInfo, Plugin>> loaded = this.loadBundles(this.seenBundles);
        pluginsLoaded.addAll(loaded);
        this.plugins = Collections.unmodifiableList(pluginsLoaded);
        List<String> list = MANDATORY_SETTING.get(settings);
        if (!list.isEmpty()) {
            HashSet<String> missingPlugins = new HashSet<String>();
            for (String mandatoryPlugin : list) {
                if (pluginsNames.contains(mandatoryPlugin) || missingPlugins.contains(mandatoryPlugin)) continue;
                missingPlugins.add(mandatoryPlugin);
            }
            if (!missingPlugins.isEmpty()) {
                String message = String.format(Locale.ROOT, "missing mandatory plugins [%s], found plugins [%s]", String.join((CharSequence)", ", missingPlugins), String.join((CharSequence)", ", pluginsNames));
                throw new IllegalStateException(message);
            }
        }
        PluginsService.logPluginInfo(pluginsList, "plugin", LOGGER);
    }

    private static void logPluginInfo(List<PluginInfo> pluginInfos, String type, Logger logger) {
        assert (pluginInfos != null);
        if (pluginInfos.isEmpty()) {
            logger.info("no " + type + "s loaded");
        } else {
            for (String name : pluginInfos.stream().map(PluginInfo::getName).sorted().collect(Collectors.toList())) {
                logger.info("loaded " + type + " [" + name + "]");
            }
        }
    }

    public Settings updatedSettings() {
        HashMap<String, String> foundSettings = new HashMap<String, String>();
        Settings.Builder builder = Settings.builder();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            Settings settings = ((Plugin)plugin.v2()).additionalSettings();
            for (String setting : settings.keySet()) {
                String oldPlugin = foundSettings.put(setting, ((PluginInfo)plugin.v1()).getName());
                if (oldPlugin == null) continue;
                throw new IllegalArgumentException("Cannot have additional setting [" + setting + "] in plugin [" + ((PluginInfo)plugin.v1()).getName() + "], already added in plugin [" + oldPlugin + "]");
            }
            builder.put(settings);
        }
        return builder.put(this.settings).build();
    }

    public Collection<Module> createGuiceModules() {
        ArrayList<Module> modules = new ArrayList<Module>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            modules.addAll(((Plugin)plugin.v2()).createGuiceModules());
        }
        return modules;
    }

    public Collection<Class<? extends LifecycleComponent>> getGuiceServiceClasses() {
        ArrayList<Class<? extends LifecycleComponent>> services = new ArrayList<Class<? extends LifecycleComponent>>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            services.addAll(((Plugin)plugin.v2()).getGuiceServiceClasses());
        }
        return services;
    }

    public void onIndexModule(IndexModule indexModule) {
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            ((Plugin)plugin.v2()).onIndexModule(indexModule);
        }
    }

    public static List<Path> findPluginDirs(Path rootPath) throws IOException {
        ArrayList<Path> plugins = new ArrayList<Path>();
        HashSet<String> seen = new HashSet<String>();
        if (Files.exists(rootPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(rootPath);){
                for (Path plugin : stream) {
                    if (FileSystemUtils.isHidden(plugin) || FileSystemUtils.isDesktopServicesStore(plugin) || plugin.getFileName().toString().startsWith(".removing-")) continue;
                    if (!seen.add(plugin.getFileName().toString())) {
                        throw new IllegalStateException("duplicate plugin: " + String.valueOf(plugin));
                    }
                    if (!Files.exists(plugin.resolve("plugin-descriptor.properties"), new LinkOption[0])) continue;
                    plugins.add(plugin);
                }
            }
        }
        return plugins;
    }

    static void checkForFailedPluginRemovals(Path pluginsDirectory) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(pluginsDirectory, ".removing-*");){
            Iterator<Path> iterator = stream.iterator();
            if (iterator.hasNext()) {
                Path removing = iterator.next();
                String fileName = removing.getFileName().toString();
                String name = fileName.substring(1 + fileName.indexOf("-"));
                String message = String.format(Locale.ROOT, "found file [%s] from a failed attempt to remove the plugin [%s]; execute [elasticsearch-plugin remove %2$s]", removing, name);
                throw new IllegalStateException(message);
            }
        }
    }

    static Set<Bundle> getPluginBundles(Path pluginsDirectory) throws IOException {
        return PluginsService.findBundles(pluginsDirectory, "plugin");
    }

    private static Set<Bundle> findBundles(Path directory, String type) throws IOException {
        HashSet<Bundle> bundles = new HashSet<Bundle>();
        for (Path plugin : PluginsService.findPluginDirs(directory)) {
            Bundle bundle = PluginsService.readPluginBundle(bundles, plugin, type);
            bundles.add(bundle);
        }
        return bundles;
    }

    private static Bundle readPluginBundle(Set<Bundle> bundles, Path plugin, String type) throws IOException {
        PluginInfo info;
        LogManager.getLogger(PluginsService.class).trace("--- adding [{}] [{}]", (Object)type, (Object)plugin.toAbsolutePath());
        try {
            info = PluginInfo.readFromProperties(plugin);
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not load plugin descriptor for " + type + " directory [" + String.valueOf(plugin.getFileName()) + "]", e);
        }
        Bundle bundle = new Bundle(info, plugin);
        if (!bundles.add(bundle)) {
            throw new IllegalStateException("duplicate " + type + ": " + String.valueOf(info));
        }
        return bundle;
    }

    static List<Bundle> sortBundles(Set<Bundle> bundles) {
        LinkedHashSet<Bundle> sortedBundles = new LinkedHashSet<Bundle>(bundles);
        return new ArrayList<Bundle>(sortedBundles);
    }

    private List<Tuple<PluginInfo, Plugin>> loadBundles(Set<Bundle> bundles) {
        ArrayList<Tuple> plugins = new ArrayList<Tuple>();
        HashMap<String, Plugin> loaded = new HashMap<String, Plugin>();
        HashMap<String, Set<URL>> transitiveUrls = new HashMap<String, Set<URL>>();
        List<Bundle> sortedBundles = PluginsService.sortBundles(bundles);
        for (Bundle bundle : sortedBundles) {
            PluginsService.checkBundleJarHell(JarHell.parseClassPath(), bundle, transitiveUrls);
            Plugin plugin = this.loadBundle(bundle, loaded);
            plugins.add(new Tuple((Object)bundle.plugin, (Object)plugin));
        }
        return Collections.unmodifiableList(plugins);
    }

    static void checkBundleJarHell(Set<URL> classpath, Bundle bundle, Map<String, Set<URL>> transitiveUrls) {
        try {
            Logger logger = LogManager.getLogger(JarHell.class);
            HashSet<URL> urls = new HashSet<URL>();
            urls.addAll(bundle.urls);
            JarHell.checkJarHell(urls, arg_0 -> ((Logger)logger).debug(arg_0));
            transitiveUrls.put(bundle.plugin.getName(), urls);
            HashSet<URL> intersection = new HashSet<URL>(classpath);
            intersection.retainAll(bundle.urls);
            if (!intersection.isEmpty()) {
                throw new IllegalStateException("jar hell! duplicate codebases between plugin and core: " + String.valueOf(intersection));
            }
            HashSet<URL> union = new HashSet<URL>(classpath);
            union.addAll(bundle.urls);
            JarHell.checkJarHell(union, arg_0 -> ((Logger)logger).debug(arg_0));
        }
        catch (Exception e) {
            throw new IllegalStateException("failed to load plugin " + bundle.plugin.getName() + " due to jar hell", e);
        }
    }

    private Plugin loadBundle(Bundle bundle, Map<String, Plugin> loaded) {
        String name = bundle.plugin.getName();
        URLClassLoader loader = URLClassLoader.newInstance(bundle.urls.toArray(new URL[0]));
        PluginsService.reloadLuceneSPI(loader);
        Class<? extends Plugin> pluginClass = this.loadPluginClass(bundle.plugin.getClassname(), loader);
        Plugin plugin = this.loadPlugin(pluginClass, this.settings, this.configPath);
        loaded.put(name, plugin);
        return plugin;
    }

    static void reloadLuceneSPI(ClassLoader loader) {
        PostingsFormat.reloadPostingsFormats((ClassLoader)loader);
        DocValuesFormat.reloadDocValuesFormats((ClassLoader)loader);
        Codec.reloadCodecs((ClassLoader)loader);
        CharFilterFactory.reloadCharFilters((ClassLoader)loader);
        TokenFilterFactory.reloadTokenFilters((ClassLoader)loader);
        TokenizerFactory.reloadTokenizers((ClassLoader)loader);
    }

    private Class<? extends Plugin> loadPluginClass(String className, ClassLoader loader) {
        try {
            return loader.loadClass(className).asSubclass(Plugin.class);
        }
        catch (ClassNotFoundException e) {
            throw new ElasticsearchException("Could not find plugin class [" + className + "]", (Throwable)e, new Object[0]);
        }
    }

    private Plugin loadPlugin(Class<? extends Plugin> pluginClass, Settings settings, Path configPath) {
        Constructor<?>[] constructors = pluginClass.getConstructors();
        if (constructors.length == 0) {
            throw new IllegalStateException("no public constructor for [" + pluginClass.getName() + "]");
        }
        if (constructors.length > 1) {
            throw new IllegalStateException("no unique public constructor for [" + pluginClass.getName() + "]");
        }
        Constructor<?> constructor = constructors[0];
        if (constructor.getParameterCount() > 2) {
            throw new IllegalStateException(this.signatureMessage(pluginClass));
        }
        Class<?>[] parameterTypes = constructor.getParameterTypes();
        try {
            if (constructor.getParameterCount() == 2 && parameterTypes[0] == Settings.class && parameterTypes[1] == Path.class) {
                return (Plugin)constructor.newInstance(settings, configPath);
            }
            if (constructor.getParameterCount() == 1 && parameterTypes[0] == Settings.class) {
                return (Plugin)constructor.newInstance(settings);
            }
            if (constructor.getParameterCount() == 0) {
                return (Plugin)constructor.newInstance(new Object[0]);
            }
            throw new IllegalStateException(this.signatureMessage(pluginClass));
        }
        catch (ReflectiveOperationException e) {
            throw new IllegalStateException("failed to load plugin class [" + pluginClass.getName() + "]", e);
        }
    }

    private String signatureMessage(Class<? extends Plugin> clazz) {
        return String.format(Locale.ROOT, "no public constructor of correct signature for [%s]; must be [%s], [%s], or [%s]", clazz.getName(), "(org.elasticsearch.common.settings.Settings,java.nio.file.Path)", "(org.elasticsearch.common.settings.Settings)", "()");
    }

    public <T> List<T> filterPlugins(Class<T> type) {
        return this.plugins.stream().filter(x -> type.isAssignableFrom(((Plugin)x.v2()).getClass())).map(p -> p.v2()).collect(Collectors.toList());
    }

    public List<ClassLoader> classLoaders() {
        ArrayList<ClassLoader> classLoaders = new ArrayList<ClassLoader>(this.seenBundles.size());
        for (Bundle bundle : this.seenBundles) {
            URLClassLoader loader = URLClassLoader.newInstance(bundle.urls.toArray(new URL[0]));
            classLoaders.add(loader);
        }
        return classLoaders;
    }

    static class Bundle {
        final PluginInfo plugin;
        final Set<URL> urls;

        Bundle(PluginInfo plugin, Path dir) throws IOException {
            this.plugin = Objects.requireNonNull(plugin);
            LinkedHashSet<URL> urls = new LinkedHashSet<URL>();
            try (DirectoryStream<Path> jarStream = Files.newDirectoryStream(dir, "*.jar");){
                for (Path jar : jarStream) {
                    URL url = jar.toRealPath(new LinkOption[0]).toUri().toURL();
                    if (urls.add(url)) continue;
                    throw new IllegalStateException("duplicate codebase: " + String.valueOf(url));
                }
            }
            this.urls = Objects.requireNonNull(urls);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Bundle bundle = (Bundle)o;
            return Objects.equals(this.plugin, bundle.plugin);
        }

        public int hashCode() {
            return Objects.hash(this.plugin);
        }
    }
}

