/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.threadpool;

import io.crate.common.unit.TimeValue;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.SizeValue;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.EsThreadPoolExecutor;
import org.elasticsearch.common.util.concurrent.XRejectedExecutionHandler;
import org.elasticsearch.node.Node;
import org.elasticsearch.threadpool.ExecutorBuilder;
import org.elasticsearch.threadpool.FixedExecutorBuilder;
import org.elasticsearch.threadpool.PrioFixedExecutorBuilder;
import org.elasticsearch.threadpool.ScalingExecutorBuilder;
import org.elasticsearch.threadpool.ScheduledCancellableAdapter;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPoolStats;
import org.jetbrains.annotations.Nullable;

public class ThreadPool
implements Scheduler {
    private static final Logger LOGGER = LogManager.getLogger(ThreadPool.class);
    private final Map<String, ExecutorHolder> executors;
    private final CachedTimeThread cachedTimeThread;
    private final List<ExecutorBuilder> builders;
    private final ScheduledThreadPoolExecutor scheduler;
    public static Setting<TimeValue> ESTIMATED_TIME_INTERVAL_SETTING = Setting.timeSetting("thread_pool.estimated_time_interval", TimeValue.timeValueMillis((long)200L), TimeValue.ZERO, Setting.Property.NodeScope);

    public Collection<ExecutorBuilder> builders() {
        return this.builders;
    }

    public ThreadPool(Settings settings) {
        assert (Node.NODE_NAME_SETTING.exists(settings));
        int availableProcessors = EsExecutors.numberOfProcessors(settings);
        int halfProcMaxAt5 = ThreadPool.halfNumberOfProcessorsMaxFive(availableProcessors);
        int halfProcMaxAt10 = ThreadPool.halfNumberOfProcessorsMaxTen(availableProcessors);
        int genericThreadPoolMax = ThreadPool.boundedBy(4 * availableProcessors, 128, 512);
        this.builders = List.of(new ScalingExecutorBuilder("generic", 4, genericThreadPoolMax, TimeValue.timeValueSeconds((long)30L)), new FixedExecutorBuilder(settings, "write", availableProcessors, 200), new PrioFixedExecutorBuilder(settings, "search", ThreadPool.searchThreadPoolSize(availableProcessors), 1000), new ScalingExecutorBuilder("management", 1, 5, TimeValue.timeValueMinutes((long)5L)), new FixedExecutorBuilder(settings, "listener", halfProcMaxAt10, -1), new ScalingExecutorBuilder("flush", 1, halfProcMaxAt5, TimeValue.timeValueMinutes((long)5L)), new ScalingExecutorBuilder("refresh", 1, halfProcMaxAt10, TimeValue.timeValueMinutes((long)5L)), new ScalingExecutorBuilder("warmer", 1, halfProcMaxAt5, TimeValue.timeValueMinutes((long)5L)), new ScalingExecutorBuilder("snapshot", 1, halfProcMaxAt5, TimeValue.timeValueMinutes((long)5L)), new ScalingExecutorBuilder("fetch_shard_started", 1, 2 * availableProcessors, TimeValue.timeValueMinutes((long)5L)), new FixedExecutorBuilder(settings, "force_merge", 1, -1), new ScalingExecutorBuilder("fetch_shard_store", 1, 2 * availableProcessors, TimeValue.timeValueMinutes((long)5L)), new FixedExecutorBuilder(settings, "logical_replication", ThreadPool.searchThreadPoolSize(availableProcessors), 100));
        HashMap<String, ExecutorHolder> executors = HashMap.newHashMap(this.builders.size() + 1);
        for (ExecutorBuilder builder : this.builders) {
            ExecutorHolder previous;
            ExecutorHolder executorHolder = builder.build(settings);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("created thread pool: {}", (Object)builder.formatInfo(executorHolder.info));
            }
            if ((previous = executors.put(executorHolder.info.name(), executorHolder)) == null) continue;
            throw new IllegalStateException("duplicate executors with name [" + executorHolder.info.name() + "] registered");
        }
        executors.put("same", new ExecutorHolder(EsExecutors.directExecutor(), new Info("same", ThreadPoolType.DIRECT)));
        this.executors = Map.copyOf(executors);
        this.scheduler = Scheduler.initScheduler(settings);
        TimeValue estimatedTimeInterval = ESTIMATED_TIME_INTERVAL_SETTING.get(settings);
        this.cachedTimeThread = new CachedTimeThread(EsExecutors.threadName(settings, "[timer]"), estimatedTimeInterval.millis());
        this.cachedTimeThread.start();
    }

    public long relativeTimeInMillis() {
        return this.cachedTimeThread.relativeTimeInMillis();
    }

    public long absoluteTimeInMillis() {
        return this.cachedTimeThread.absoluteTimeInMillis();
    }

    public ThreadPoolStats stats() {
        ArrayList<ThreadPoolStats.Stats> stats = new ArrayList<ThreadPoolStats.Stats>(this.executors.size() - 1);
        for (ExecutorHolder holder : this.executors.values()) {
            ThreadPoolStats.Stats poolStats;
            String name = holder.info.name();
            if ("same".equals(name) || (poolStats = this.stats(name)) == null) continue;
            stats.add(poolStats);
        }
        return new ThreadPoolStats(stats);
    }

    @Nullable
    public ThreadPoolStats.Stats stats(String name) {
        ExecutorHolder holder = this.executors.get(name);
        if (holder == null) {
            return null;
        }
        Executor executor = holder.executor();
        if (executor instanceof ThreadPoolExecutor) {
            long l;
            ThreadPoolExecutor threadPoolExecutor = (ThreadPoolExecutor)executor;
            int n = threadPoolExecutor.getPoolSize();
            int n2 = threadPoolExecutor.getQueue().size();
            int n3 = threadPoolExecutor.getActiveCount();
            RejectedExecutionHandler rejectedExecutionHandler = threadPoolExecutor.getRejectedExecutionHandler();
            if (rejectedExecutionHandler instanceof XRejectedExecutionHandler) {
                XRejectedExecutionHandler rejectedHandler = (XRejectedExecutionHandler)rejectedExecutionHandler;
                l = rejectedHandler.rejected();
            } else {
                l = -1L;
            }
            return new ThreadPoolStats.Stats(name, n, n2, n3, l, threadPoolExecutor.getLargestPoolSize(), threadPoolExecutor.getCompletedTaskCount());
        }
        return new ThreadPoolStats.Stats(name, -1, -1, -1, -1L, -1, -1L);
    }

    public ExecutorService generic() {
        return (ExecutorService)this.executor("generic");
    }

    public Executor executor(String name) {
        ExecutorHolder holder = this.executors.get(name);
        if (holder == null) {
            throw new IllegalArgumentException("no executor service found for [" + name + "]");
        }
        return holder.executor();
    }

    @Override
    public Scheduler.ScheduledCancellable schedule(Runnable command, TimeValue delay, String executor) {
        if (!"same".equals(executor)) {
            command = new ThreadedRunnable(command, this.executor(executor));
        }
        return new ScheduledCancellableAdapter(this.scheduler.schedule(command, delay.millis(), TimeUnit.MILLISECONDS));
    }

    public Scheduler.Cancellable scheduleUnlessShuttingDown(TimeValue delay, String executor, Runnable command) {
        try {
            return this.schedule(command, delay, executor);
        }
        catch (EsRejectedExecutionException e) {
            if (e.isExecutorShutdown()) {
                LOGGER.debug((Message)new ParameterizedMessage("could not schedule execution of [{}] after [{}] on [{}] as executor is shut down", new Object[]{command, delay, executor}), (Throwable)e);
                return Scheduler.Cancellable.CANCELLED_NOOP;
            }
            throw e;
        }
    }

    @Override
    public Scheduler.Cancellable scheduleWithFixedDelay(Runnable command, TimeValue interval, String executor) {
        return new Scheduler.ReschedulingRunnable(command, interval, executor, this, e -> {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(() -> new ParameterizedMessage("scheduled task [{}] was rejected on thread pool [{}]", (Object)command, (Object)executor), (Throwable)e);
            }
        }, e -> LOGGER.warn(() -> new ParameterizedMessage("failed to run scheduled task [{}] on thread pool [{}]", (Object)command, (Object)executor), (Throwable)e));
    }

    protected final void stopCachedTimeThread() {
        this.cachedTimeThread.running = false;
        this.cachedTimeThread.interrupt();
    }

    public void shutdown() {
        this.stopCachedTimeThread();
        this.scheduler.shutdown();
        for (ExecutorHolder executorHolder : this.executors.values()) {
            Executor executor = executorHolder.executor();
            if (!(executor instanceof ThreadPoolExecutor)) continue;
            ((ThreadPoolExecutor)executor).shutdown();
        }
    }

    public void shutdownNow() {
        this.stopCachedTimeThread();
        this.scheduler.shutdownNow();
        for (ExecutorHolder executorHolder : this.executors.values()) {
            Executor executor = executorHolder.executor();
            if (!(executor instanceof ThreadPoolExecutor)) continue;
            ((ThreadPoolExecutor)executor).shutdownNow();
        }
    }

    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        boolean result = this.scheduler.awaitTermination(timeout, unit);
        for (ExecutorHolder executorHolder : this.executors.values()) {
            Executor executor = executorHolder.executor();
            if (!(executor instanceof ThreadPoolExecutor)) continue;
            result &= ((ThreadPoolExecutor)executor).awaitTermination(timeout, unit);
        }
        this.cachedTimeThread.join(unit.toMillis(timeout));
        return result;
    }

    public ScheduledExecutorService scheduler() {
        return this.scheduler;
    }

    static int boundedBy(int value, int min, int max) {
        return Math.min(max, Math.max(min, value));
    }

    static int halfNumberOfProcessorsMaxFive(int numberOfProcessors) {
        return ThreadPool.boundedBy((numberOfProcessors + 1) / 2, 1, 5);
    }

    static int halfNumberOfProcessorsMaxTen(int numberOfProcessors) {
        return ThreadPool.boundedBy((numberOfProcessors + 1) / 2, 1, 10);
    }

    public static int searchThreadPoolSize(int availableProcessors) {
        return availableProcessors * 3 / 2 + 1;
    }

    public static boolean terminate(ExecutorService service, long timeout, TimeUnit timeUnit) {
        if (service != null) {
            service.shutdown();
            if (ThreadPool.awaitTermination(service, timeout, timeUnit)) {
                return true;
            }
            service.shutdownNow();
            return ThreadPool.awaitTermination(service, timeout, timeUnit);
        }
        return false;
    }

    private static boolean awaitTermination(ExecutorService service, long timeout, TimeUnit timeUnit) {
        try {
            if (service.awaitTermination(timeout, timeUnit)) {
                return true;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }

    public static boolean terminate(ThreadPool pool, long timeout, TimeUnit timeUnit) {
        if (pool != null) {
            pool.shutdown();
            if (ThreadPool.awaitTermination(pool, timeout, timeUnit)) {
                return true;
            }
            pool.shutdownNow();
            return ThreadPool.awaitTermination(pool, timeout, timeUnit);
        }
        return false;
    }

    private static boolean awaitTermination(ThreadPool threadPool, long timeout, TimeUnit timeUnit) {
        try {
            if (threadPool.awaitTermination(timeout, timeUnit)) {
                return true;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }

    public static boolean assertNotScheduleThread(String reason) {
        assert (!Thread.currentThread().getName().contains("scheduler")) : "Expected current thread [" + String.valueOf(Thread.currentThread()) + "] to not be the scheduler thread. Reason: [" + reason + "]";
        return true;
    }

    public static boolean assertCurrentMethodIsNotCalledRecursively() {
        StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
        assert (stackTraceElements.length >= 3) : stackTraceElements.length;
        assert (stackTraceElements[0].getMethodName().equals("getStackTrace")) : stackTraceElements[0];
        assert (stackTraceElements[1].getMethodName().equals("assertCurrentMethodIsNotCalledRecursively")) : stackTraceElements[1];
        StackTraceElement testingMethod = stackTraceElements[2];
        for (int i = 3; i < stackTraceElements.length; ++i) {
            assert (!stackTraceElements[i].getClassName().equals(testingMethod.getClassName()) || !stackTraceElements[i].getMethodName().equals(testingMethod.getMethodName())) : testingMethod.getClassName() + "#" + testingMethod.getMethodName() + " is called recursively";
        }
        return true;
    }

    public static class Names {
        public static final String SAME = "same";
        public static final String GENERIC = "generic";
        public static final String LISTENER = "listener";
        public static final String WRITE = "write";
        public static final String SEARCH = "search";
        public static final String MANAGEMENT = "management";
        public static final String FLUSH = "flush";
        public static final String REFRESH = "refresh";
        public static final String WARMER = "warmer";
        public static final String SNAPSHOT = "snapshot";
        public static final String FORCE_MERGE = "force_merge";
        public static final String FETCH_SHARD_STARTED = "fetch_shard_started";
        public static final String FETCH_SHARD_STORE = "fetch_shard_store";
        public static final String LOGICAL_REPLICATION = "logical_replication";
    }

    record ExecutorHolder(Executor executor, Info info) {
        ExecutorHolder {
            assert (executor instanceof EsThreadPoolExecutor || executor == EsExecutors.directExecutor()) : "Executor must either be the DIRECT_EXECUTOR or an instance of EsThreadPoolExecutor";
        }
    }

    public record Info(String name, ThreadPoolType type, int min, int max, @Nullable TimeValue keepAlive, @Nullable SizeValue queueSize) implements Writeable
    {
        public Info(String name, ThreadPoolType type) {
            this(name, type, -1, -1, null, null);
        }

        public static Info of(StreamInput in) throws IOException {
            String name = in.readString();
            ThreadPoolType type = ThreadPoolType.fromType(in.readString());
            int min = in.readInt();
            int max = in.readInt();
            TimeValue keepAlive = in.readOptionalTimeValue();
            SizeValue queueSize = in.readOptionalWriteable(SizeValue::new);
            return new Info(name, type, min, max, keepAlive, queueSize);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.name);
            out.writeString(this.type.getType());
            out.writeInt(this.min);
            out.writeInt(this.max);
            out.writeOptionalTimeValue(this.keepAlive);
            out.writeOptionalWriteable(this.queueSize);
        }
    }

    public static enum ThreadPoolType {
        DIRECT("direct"),
        FIXED("fixed"),
        SCALING("scaling");

        private final String type;
        private static final Map<String, ThreadPoolType> TYPE_MAP;

        public String getType() {
            return this.type;
        }

        private ThreadPoolType(String type) {
            this.type = type;
        }

        public static ThreadPoolType fromType(String type) {
            ThreadPoolType threadPoolType = TYPE_MAP.get(type);
            if (threadPoolType == null) {
                throw new IllegalArgumentException("no ThreadPoolType for " + type);
            }
            return threadPoolType;
        }

        static {
            HashMap<String, ThreadPoolType> typeMap = new HashMap<String, ThreadPoolType>();
            for (ThreadPoolType threadPoolType : ThreadPoolType.values()) {
                typeMap.put(threadPoolType.getType(), threadPoolType);
            }
            TYPE_MAP = Collections.unmodifiableMap(typeMap);
        }
    }

    static class CachedTimeThread
    extends Thread {
        final long interval;
        volatile boolean running = true;
        volatile long relativeMillis;
        volatile long absoluteMillis;

        CachedTimeThread(String name, long interval) {
            super(name);
            this.interval = interval;
            this.relativeMillis = TimeValue.nsecToMSec((long)System.nanoTime());
            this.absoluteMillis = System.currentTimeMillis();
            this.setDaemon(true);
        }

        long relativeTimeInMillis() {
            if (0L < this.interval) {
                return this.relativeMillis;
            }
            return TimeValue.nsecToMSec((long)System.nanoTime());
        }

        long absoluteTimeInMillis() {
            if (0L < this.interval) {
                return this.absoluteMillis;
            }
            return System.currentTimeMillis();
        }

        @Override
        public void run() {
            while (this.running && 0L < this.interval) {
                this.relativeMillis = TimeValue.nsecToMSec((long)System.nanoTime());
                this.absoluteMillis = System.currentTimeMillis();
                try {
                    Thread.sleep(this.interval);
                }
                catch (InterruptedException e) {
                    this.running = false;
                    return;
                }
            }
        }
    }

    static class ThreadedRunnable
    implements Runnable {
        private final Runnable runnable;
        private final Executor executor;

        ThreadedRunnable(Runnable runnable, Executor executor) {
            this.runnable = runnable;
            this.executor = executor;
        }

        @Override
        public void run() {
            try {
                this.executor.execute(this.runnable);
            }
            catch (EsRejectedExecutionException e) {
                if (e.isExecutorShutdown()) {
                    LOGGER.debug((Message)new ParameterizedMessage("could not schedule execution of [{}] on [{}] as executor is shut down", (Object)this.runnable, (Object)this.executor), (Throwable)e);
                }
                throw e;
            }
        }

        public int hashCode() {
            return this.runnable.hashCode();
        }

        public boolean equals(Object obj) {
            return obj instanceof ThreadedRunnable && this.runnable.equals(obj);
        }

        public String toString() {
            return "[threaded] " + this.runnable.toString();
        }
    }
}

