/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.shard;

import io.crate.common.io.IOUtils;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Strings;
import org.elasticsearch.common.io.stream.NamedWriteableRegistry;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardStateMetadata;

public final class ShardPath {
    public static final String INDEX_FOLDER_NAME = "index";
    public static final String TRANSLOG_FOLDER_NAME = "translog";
    private final Path path;
    private final ShardId shardId;
    private final Path shardStatePath;
    private final boolean isCustomDataPath;

    public ShardPath(boolean isCustomDataPath, Path dataPath, Path shardStatePath, ShardId shardId) {
        assert (dataPath.getFileName().toString().equals(Integer.toString(shardId.id()))) : "dataPath must end with the shard ID but didn't: " + dataPath.toString();
        assert (shardStatePath.getFileName().toString().equals(Integer.toString(shardId.id()))) : "shardStatePath must end with the shard ID but didn't: " + dataPath.toString();
        assert (dataPath.getParent().getFileName().toString().equals(shardId.getIndex().getUUID())) : "dataPath must end with index path id but didn't: " + dataPath.toString();
        assert (shardStatePath.getParent().getFileName().toString().equals(shardId.getIndex().getUUID())) : "shardStatePath must end with index path id but didn't: " + dataPath.toString();
        if (isCustomDataPath && dataPath.equals(shardStatePath)) {
            throw new IllegalArgumentException("shard state path must be different to the data path when using custom data paths");
        }
        this.isCustomDataPath = isCustomDataPath;
        this.path = dataPath;
        this.shardId = shardId;
        this.shardStatePath = shardStatePath;
    }

    public Path resolveTranslog() {
        return this.path.resolve(TRANSLOG_FOLDER_NAME);
    }

    public Path resolveIndex() {
        return this.path.resolve(INDEX_FOLDER_NAME);
    }

    public Path getDataPath() {
        return this.path;
    }

    public boolean exists() {
        return Files.exists(this.path, new LinkOption[0]);
    }

    public ShardId getShardId() {
        return this.shardId;
    }

    public Path getShardStatePath() {
        return this.shardStatePath;
    }

    public Path getRootDataPath() {
        Path noIndexShardId = this.getDataPath().getParent().getParent();
        return this.isCustomDataPath ? noIndexShardId : noIndexShardId.getParent();
    }

    public Path getRootStatePath() {
        return this.getShardStatePath().getParent().getParent().getParent();
    }

    public boolean isCustomDataPath() {
        return this.isCustomDataPath;
    }

    public static ShardPath loadShardPath(Logger logger, NodeEnvironment env, ShardId shardId, String customDataPath) throws IOException {
        Path[] paths = env.availableShardPaths(shardId);
        Path sharedDataPath = env.sharedDataPath();
        return ShardPath.loadShardPath(logger, shardId, customDataPath, paths, sharedDataPath);
    }

    public static ShardPath loadShardPath(Logger logger, ShardId shardId, String customDataPath, Path[] availableShardPaths, Path sharedDataPath) throws IOException {
        String indexUUID = shardId.getIndex().getUUID();
        Path loadedPath = null;
        for (Path path : availableShardPaths) {
            ShardStateMetadata load = ShardStateMetadata.FORMAT.loadLatestState(logger, NamedWriteableRegistry.EMPTY, NamedXContentRegistry.EMPTY, path);
            if (load == null) continue;
            if (!load.indexUUID.equals(indexUUID) && !"_na_".equals(load.indexUUID)) {
                logger.warn("{} found shard on path: [{}] with a different index UUID - this shard seems to be leftover from a different index with the same name. Remove the leftover shard in order to reuse the path with the current index", (Object)shardId, (Object)path);
                throw new IllegalStateException(String.valueOf(shardId) + " index UUID in shard state was: " + load.indexUUID + " expected: " + indexUUID + " on shard path: " + String.valueOf(path));
            }
            if (loadedPath == null) {
                loadedPath = path;
                continue;
            }
            throw new IllegalStateException(String.valueOf(shardId) + " more than one shard state found");
        }
        if (loadedPath == null) {
            return null;
        }
        Path statePath = loadedPath;
        boolean hasCustomDataPath = Strings.isNotEmpty((CharSequence)customDataPath);
        Path dataPath = hasCustomDataPath ? NodeEnvironment.resolveCustomLocation(customDataPath, shardId, sharedDataPath) : statePath;
        logger.debug("{} loaded data path [{}], state path [{}]", (Object)shardId, (Object)dataPath, (Object)statePath);
        return new ShardPath(hasCustomDataPath, dataPath, statePath, shardId);
    }

    public static void deleteLeftoverShardDirectory(Logger logger, NodeEnvironment env, ShardLock lock, IndexSettings indexSettings) throws IOException {
        Path[] paths;
        String indexUUID = indexSettings.getUUID();
        for (Path path : paths = env.availableShardPaths(lock.getShardId())) {
            ShardStateMetadata load = ShardStateMetadata.FORMAT.loadLatestState(logger, NamedWriteableRegistry.EMPTY, NamedXContentRegistry.EMPTY, path);
            if (load == null || load.indexUUID.equals(indexUUID) || "_na_".equals(load.indexUUID)) continue;
            logger.warn("{} deleting leftover shard on path: [{}] with a different index UUID", (Object)lock.getShardId(), (Object)path);
            assert (Files.isDirectory(path, new LinkOption[0])) : String.valueOf(path) + " is not a directory";
            NodeEnvironment.acquireFSLockForPaths(indexSettings, paths);
            IOUtils.rm((Path[])new Path[]{path});
        }
    }

    public static ShardPath selectNewPathForShard(NodeEnvironment env, ShardId shardId, IndexSettings indexSettings, long avgShardSizeInBytes, Map<Path, Integer> dataPathToShardCount) throws IOException {
        Path statePath;
        Path dataPath;
        if (indexSettings.hasCustomDataPath()) {
            dataPath = env.resolveCustomLocation(indexSettings.customDataPath(), shardId);
            statePath = env.nodePaths()[0].resolve(shardId);
        } else {
            BigInteger totFreeSpace = BigInteger.ZERO;
            for (NodeEnvironment.NodePath nodePath : env.nodePaths()) {
                totFreeSpace = totFreeSpace.add(BigInteger.valueOf(nodePath.fileStore.getUsableSpace()));
            }
            BigInteger estShardSizeInBytes = BigInteger.valueOf(avgShardSizeInBytes).max(totFreeSpace.divide(BigInteger.valueOf(20L)));
            NodeEnvironment.NodePath[] paths = env.nodePaths();
            NodeEnvironment.NodePath bestPath = ShardPath.getPathWithMostFreeSpace(env);
            if (paths.length != 1) {
                Map<NodeEnvironment.NodePath, Long> pathToShardCount = env.shardCountPerPath(shardId.getIndex());
                HashMap<NodeEnvironment.NodePath, BigInteger> pathsToSpace = new HashMap<NodeEnvironment.NodePath, BigInteger>(paths.length);
                for (NodeEnvironment.NodePath nodePath : paths) {
                    FileStore fileStore = nodePath.fileStore;
                    BigInteger usableBytes = BigInteger.valueOf(fileStore.getUsableSpace());
                    pathsToSpace.put(nodePath, usableBytes);
                }
                bestPath = Arrays.stream(paths).filter(path -> ((BigInteger)pathsToSpace.get(path)).subtract(estShardSizeInBytes).compareTo(BigInteger.ZERO) > 0).sorted((p1, p2) -> {
                    int cmp = Long.compare(pathToShardCount.getOrDefault(p1, 0L), pathToShardCount.getOrDefault(p2, 0L));
                    if (cmp == 0 && (cmp = Integer.compare(dataPathToShardCount.getOrDefault(p1.path, 0), dataPathToShardCount.getOrDefault(p2.path, 0))) == 0) {
                        cmp = ((BigInteger)pathsToSpace.get(p2)).compareTo((BigInteger)pathsToSpace.get(p1));
                    }
                    return cmp;
                }).findFirst().orElse(bestPath);
            }
            dataPath = statePath = bestPath.resolve(shardId);
        }
        return new ShardPath(indexSettings.hasCustomDataPath(), dataPath, statePath, shardId);
    }

    static NodeEnvironment.NodePath getPathWithMostFreeSpace(NodeEnvironment env) throws IOException {
        NodeEnvironment.NodePath[] paths = env.nodePaths();
        NodeEnvironment.NodePath bestPath = null;
        long maxUsableBytes = Long.MIN_VALUE;
        for (NodeEnvironment.NodePath nodePath : paths) {
            FileStore fileStore = nodePath.fileStore;
            long usableBytes = fileStore.getUsableSpace();
            assert (usableBytes >= 0L) : "usable bytes must be >= 0, got: " + usableBytes;
            if (bestPath != null && usableBytes <= maxUsableBytes) continue;
            maxUsableBytes = usableBytes;
            bestPath = nodePath;
        }
        return bestPath;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ShardPath shardPath = (ShardPath)o;
        if (!Objects.equals(this.shardId, shardPath.shardId)) {
            return false;
        }
        return Objects.equals(this.path, shardPath.path);
    }

    public int hashCode() {
        int result = this.path != null ? this.path.hashCode() : 0;
        result = 31 * result + (this.shardId != null ? this.shardId.hashCode() : 0);
        return result;
    }

    public String toString() {
        return "ShardPath{path=" + String.valueOf(this.path) + ", shard=" + String.valueOf(this.shardId) + "}";
    }
}

