/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices;

import io.crate.common.unit.TimeValue;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.RejectableRunnable;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.IndexingOperationListener;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryController
implements IndexingOperationListener,
Closeable {
    private static final Logger LOGGER = LogManager.getLogger(IndexingMemoryController.class);
    public static final Setting<ByteSizeValue> INDEX_BUFFER_SIZE_SETTING = Setting.memorySizeSetting("indices.memory.index_buffer_size", "10%", Setting.Property.NodeScope);
    public static final Setting<ByteSizeValue> MIN_INDEX_BUFFER_SIZE_SETTING = Setting.byteSizeSetting("indices.memory.min_index_buffer_size", new ByteSizeValue(48L, ByteSizeUnit.MB), new ByteSizeValue(0L, ByteSizeUnit.BYTES), new ByteSizeValue(Long.MAX_VALUE, ByteSizeUnit.BYTES), Setting.Property.NodeScope);
    public static final Setting<ByteSizeValue> MAX_INDEX_BUFFER_SIZE_SETTING = Setting.byteSizeSetting("indices.memory.max_index_buffer_size", new ByteSizeValue(-1L), new ByteSizeValue(-1L), new ByteSizeValue(Long.MAX_VALUE, ByteSizeUnit.BYTES), Setting.Property.NodeScope);
    public static final Setting<TimeValue> SHARD_INACTIVE_TIME_SETTING = Setting.positiveTimeSetting("indices.memory.shard_inactive_time", TimeValue.timeValueMinutes((long)5L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> SHARD_MEMORY_INTERVAL_TIME_SETTING = Setting.positiveTimeSetting("indices.memory.interval", TimeValue.timeValueSeconds((long)5L), Setting.Property.NodeScope);
    private final ThreadPool threadPool;
    private final Iterable<IndexShard> indexShards;
    private final ByteSizeValue indexingBuffer;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private final Set<IndexShard> throttled = new HashSet<IndexShard>();
    private final Scheduler.Cancellable scheduler;
    private static final EnumSet<IndexShardState> CAN_WRITE_INDEX_BUFFER_STATES = EnumSet.of(IndexShardState.RECOVERING, IndexShardState.POST_RECOVERY, IndexShardState.STARTED);
    private final ShardsIndicesStatusChecker statusChecker;

    IndexingMemoryController(Settings settings, ThreadPool threadPool, Iterable<IndexShard> indexServices) {
        this.indexShards = indexServices;
        ByteSizeValue indexingBuffer = INDEX_BUFFER_SIZE_SETTING.get(settings);
        String indexingBufferSetting = settings.get(INDEX_BUFFER_SIZE_SETTING.getKey());
        if (indexingBufferSetting == null || indexingBufferSetting.endsWith("%")) {
            ByteSizeValue minIndexingBuffer = MIN_INDEX_BUFFER_SIZE_SETTING.get(settings);
            ByteSizeValue maxIndexingBuffer = MAX_INDEX_BUFFER_SIZE_SETTING.get(settings);
            if (indexingBuffer.getBytes() < minIndexingBuffer.getBytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer.getBytes() != -1L && indexingBuffer.getBytes() > maxIndexingBuffer.getBytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        }
        this.indexingBuffer = indexingBuffer;
        this.inactiveTime = SHARD_INACTIVE_TIME_SETTING.get(settings);
        this.interval = SHARD_MEMORY_INTERVAL_TIME_SETTING.get(settings);
        this.statusChecker = new ShardsIndicesStatusChecker();
        LOGGER.debug("using indexing buffer size [{}] with {} [{}], {} [{}]", (Object)this.indexingBuffer, (Object)SHARD_INACTIVE_TIME_SETTING.getKey(), (Object)this.inactiveTime, (Object)SHARD_MEMORY_INTERVAL_TIME_SETTING.getKey(), (Object)this.interval);
        this.scheduler = this.scheduleTask(threadPool);
        this.threadPool = threadPool;
    }

    protected Scheduler.Cancellable scheduleTask(ThreadPool threadPool) {
        return threadPool.scheduleWithFixedDelay(this.statusChecker, this.interval, "same");
    }

    @Override
    public void close() {
        this.scheduler.cancel();
    }

    ByteSizeValue indexingBufferSize() {
        return this.indexingBuffer;
    }

    protected List<IndexShard> availableShards() {
        ArrayList<IndexShard> availableShards = new ArrayList<IndexShard>();
        for (IndexShard shard : this.indexShards) {
            if (!CAN_WRITE_INDEX_BUFFER_STATES.contains((Object)shard.state())) continue;
            availableShards.add(shard);
        }
        return availableShards;
    }

    protected long getIndexBufferRAMBytesUsed(IndexShard shard) {
        return shard.getIndexBufferRAMBytesUsed();
    }

    protected long getShardWritingBytes(IndexShard shard) {
        return shard.getWritingBytes();
    }

    protected void writeIndexingBufferAsync(final IndexShard shard) {
        this.threadPool.executor("refresh").execute(new RejectableRunnable(){

            @Override
            public void doRun() {
                shard.writeIndexingBuffer();
            }

            @Override
            public void onFailure(Exception e) {
                LOGGER.warn(() -> new ParameterizedMessage("failed to write indexing buffer for shard [{}]; ignoring", (Object)shard.shardId()), (Throwable)e);
            }
        });
    }

    void forceCheck() {
        this.statusChecker.run();
    }

    protected void activateThrottling(IndexShard shard) {
        shard.activateThrottling();
    }

    protected void deactivateThrottling(IndexShard shard) {
        shard.deactivateThrottling();
    }

    @Override
    public void postIndex(ShardId shardId, Engine.Index index, Engine.IndexResult result) {
        this.recordOperationBytes(index, result);
    }

    @Override
    public void postDelete(ShardId shardId, Engine.Delete delete, Engine.DeleteResult result) {
        this.recordOperationBytes(delete, result);
    }

    private void recordOperationBytes(Engine.Operation operation, Engine.Result result) {
        if (result.getResultType() == Engine.Result.Type.SUCCESS) {
            this.statusChecker.bytesWritten(operation.estimatedSizeInBytes());
        }
    }

    protected void checkIdle(IndexShard shard, long inactiveTimeNS) {
        try {
            shard.flushOnIdle(inactiveTimeNS);
        }
        catch (AlreadyClosedException e) {
            LOGGER.trace(() -> new ParameterizedMessage("ignore exception while checking if shard {} is inactive", (Object)shard.shardId()), (Throwable)e);
        }
    }

    final class ShardsIndicesStatusChecker
    implements Runnable {
        final AtomicLong bytesWrittenSinceCheck = new AtomicLong();
        final ReentrantLock runLock = new ReentrantLock();

        ShardsIndicesStatusChecker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void bytesWritten(int bytes) {
            long totalBytes = this.bytesWrittenSinceCheck.addAndGet(bytes);
            assert (totalBytes >= 0L);
            while (totalBytes > IndexingMemoryController.this.indexingBuffer.getBytes() / 30L && this.runLock.tryLock()) {
                try {
                    totalBytes = this.bytesWrittenSinceCheck.get();
                    if (totalBytes > IndexingMemoryController.this.indexingBuffer.getBytes() / 30L) {
                        this.bytesWrittenSinceCheck.addAndGet(-totalBytes);
                        this.runUnlocked();
                    }
                }
                finally {
                    this.runLock.unlock();
                }
                totalBytes = this.bytesWrittenSinceCheck.get();
            }
        }

        @Override
        public void run() {
            this.runLock.lock();
            try {
                this.runUnlocked();
            }
            finally {
                this.runLock.unlock();
            }
        }

        private void runUnlocked() {
            boolean doThrottle;
            long totalBytesUsed = 0L;
            long totalBytesWriting = 0L;
            List<IndexShard> availableShards = IndexingMemoryController.this.availableShards();
            for (IndexShard shard : availableShards) {
                IndexingMemoryController.this.checkIdle(shard, IndexingMemoryController.this.inactiveTime.nanos());
                long shardWritingBytes = IndexingMemoryController.this.getShardWritingBytes(shard);
                long shardBytesUsed = IndexingMemoryController.this.getIndexBufferRAMBytesUsed(shard);
                totalBytesWriting += shardWritingBytes;
                if ((shardBytesUsed -= shardWritingBytes) < 0L) continue;
                totalBytesUsed += shardBytesUsed;
            }
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("total indexing heap bytes used [{}] vs {} [{}], currently writing bytes [{}]", (Object)new ByteSizeValue(totalBytesUsed), (Object)INDEX_BUFFER_SIZE_SETTING.getKey(), (Object)IndexingMemoryController.this.indexingBuffer, (Object)new ByteSizeValue(totalBytesWriting));
            }
            boolean bl = doThrottle = (double)(totalBytesWriting + totalBytesUsed) > 1.5 * (double)IndexingMemoryController.this.indexingBuffer.getBytes();
            if (totalBytesUsed > IndexingMemoryController.this.indexingBuffer.getBytes()) {
                PriorityQueue<ShardAndBytesUsed> queue = new PriorityQueue<ShardAndBytesUsed>();
                for (IndexShard shard : availableShards) {
                    long shardWritingBytes = IndexingMemoryController.this.getShardWritingBytes(shard);
                    long shardBytesUsed = IndexingMemoryController.this.getIndexBufferRAMBytesUsed(shard);
                    if ((shardBytesUsed -= shardWritingBytes) < 0L || shardBytesUsed <= 0L) continue;
                    if (LOGGER.isTraceEnabled()) {
                        if (shardWritingBytes != 0L) {
                            LOGGER.trace("shard [{}] is using [{}] heap, writing [{}] heap", (Object)shard.shardId(), (Object)shardBytesUsed, (Object)shardWritingBytes);
                        } else {
                            LOGGER.trace("shard [{}] is using [{}] heap, not writing any bytes", (Object)shard.shardId(), (Object)shardBytesUsed);
                        }
                    }
                    queue.add(new ShardAndBytesUsed(shardBytesUsed, shard));
                }
                LOGGER.debug("now write some indexing buffers: total indexing heap bytes used [{}] vs {} [{}], currently writing bytes [{}], [{}] shards with non-zero indexing buffer", (Object)new ByteSizeValue(totalBytesUsed), (Object)INDEX_BUFFER_SIZE_SETTING.getKey(), (Object)IndexingMemoryController.this.indexingBuffer, (Object)new ByteSizeValue(totalBytesWriting), (Object)queue.size());
                while (totalBytesUsed > IndexingMemoryController.this.indexingBuffer.getBytes() && !queue.isEmpty()) {
                    ShardAndBytesUsed largest = (ShardAndBytesUsed)queue.poll();
                    LOGGER.debug("write indexing buffer to disk for shard [{}] to free up its [{}] indexing buffer", (Object)largest.shard.shardId(), (Object)new ByteSizeValue(largest.bytesUsed));
                    IndexingMemoryController.this.writeIndexingBufferAsync(largest.shard);
                    totalBytesUsed -= largest.bytesUsed;
                    if (!doThrottle || IndexingMemoryController.this.throttled.contains(largest.shard)) continue;
                    LOGGER.info("now throttling indexing for shard [{}]: segment writing can't keep up", (Object)largest.shard.shardId());
                    IndexingMemoryController.this.throttled.add(largest.shard);
                    IndexingMemoryController.this.activateThrottling(largest.shard);
                }
            }
            if (!doThrottle) {
                for (IndexShard shard : IndexingMemoryController.this.throttled) {
                    LOGGER.info("stop throttling indexing for shard [{}]", (Object)shard.shardId());
                    IndexingMemoryController.this.deactivateThrottling(shard);
                }
                IndexingMemoryController.this.throttled.clear();
            }
        }
    }

    private static final class ShardAndBytesUsed
    implements Comparable<ShardAndBytesUsed> {
        final long bytesUsed;
        final IndexShard shard;

        ShardAndBytesUsed(long bytesUsed, IndexShard shard) {
            this.bytesUsed = bytesUsed;
            this.shard = shard;
        }

        @Override
        public int compareTo(ShardAndBytesUsed other) {
            return Long.compare(other.bytesUsed, this.bytesUsed);
        }
    }
}

