/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import io.crate.common.io.IOUtils;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobStore;

class S3RetryingInputStream
extends InputStream {
    private static final Logger LOGGER = LogManager.getLogger(S3RetryingInputStream.class);
    static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final S3BlobStore blobStore;
    private final String blobKey;
    private final long start;
    private final long end;
    private final int maxAttempts;
    private InputStream currentStream;
    private int attempt = 1;
    private List<IOException> failures = new ArrayList<IOException>(10);
    private long currentOffset;
    private boolean closed;

    S3RetryingInputStream(S3BlobStore blobStore, String blobKey) throws IOException {
        this(blobStore, blobKey, 0L, 0x7FFFFFFFFFFFFFFEL);
    }

    S3RetryingInputStream(S3BlobStore blobStore, String blobKey, long start, long end) throws IOException {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.blobStore = blobStore;
        this.blobKey = blobKey;
        this.maxAttempts = blobStore.getMaxRetries() + 1;
        this.start = start;
        this.end = end;
        this.currentStream = this.openStream();
    }

    private InputStream openStream() throws IOException {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            GetObjectRequest getObjectRequest = new GetObjectRequest(this.blobStore.bucket(), this.blobKey);
            if (this.currentOffset > 0L || this.start > 0L || this.end < 0x7FFFFFFFFFFFFFFEL) {
                assert (this.start + this.currentOffset <= this.end) : "requesting beyond end, start = " + this.start + " offset=" + this.currentOffset + " end=" + this.end;
                getObjectRequest.setRange(Math.addExact(this.start, this.currentOffset), this.end);
            }
            S3Object s3Object = clientReference.client().getObject(getObjectRequest);
            S3ObjectInputStream s3ObjectInputStream = s3Object.getObjectContent();
            if (clientReference != null) {
                clientReference.close();
            }
            return s3ObjectInputStream;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (AmazonClientException e) {
                if (e instanceof AmazonS3Exception && 404 == ((AmazonS3Exception)e).getStatusCode()) {
                    throw this.addSuppressedExceptions(new NoSuchFileException("Blob object [" + this.blobKey + "] not found: " + e.getMessage()));
                }
                throw this.addSuppressedExceptions(e);
            }
        }
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int result = this.currentStream.read();
                ++this.currentOffset;
                return result;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead == -1) {
                    return -1;
                }
                this.currentOffset += (long)bytesRead;
                return bytesRead;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using S3RetryingInputStream after close";
            throw new IllegalStateException("using S3RetryingInputStream after close");
        }
    }

    private void reopenStreamOrFail(IOException e) throws IOException {
        if (this.attempt >= this.maxAttempts) {
            LOGGER.debug((Message)new ParameterizedMessage("failed reading [{}/{}] at offset [{}], attempt [{}] of [{}], giving up", new Object[]{this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.attempt, this.maxAttempts}), (Throwable)e);
            throw this.addSuppressedExceptions(e);
        }
        LOGGER.debug((Message)new ParameterizedMessage("failed reading [{}/{}] at offset [{}], attempt [{}] of [{}], retrying", new Object[]{this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.attempt, this.maxAttempts}), (Throwable)e);
        ++this.attempt;
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        try {
            Streams.consumeFully((InputStream)this.currentStream);
        }
        catch (Exception e2) {
            LOGGER.trace("Failed to fully consume stream on close", (Throwable)e);
        }
        IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.currentStream});
        this.currentStream = this.openStream();
    }

    @Override
    public void close() throws IOException {
        try {
            Streams.consumeFully((InputStream)this.currentStream);
        }
        catch (Exception e) {
            LOGGER.trace("Failed to fully consume stream on close", (Throwable)e);
        }
        this.currentStream.close();
        this.closed = true;
    }

    @Override
    public long skip(long n) {
        throw new UnsupportedOperationException("S3RetryingInputStream does not support seeking");
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("S3RetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (IOException failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }
}

